(ns burningswell.kafka.consumer
  (:require [burningswell.kafka.core :as core]
            [burningswell.kafka.deserializer :as deserializer]
            [peripheral.core :refer [defcomponent]])
  (:import org.apache.kafka.clients.consumer.KafkaConsumer))

(defn config
  "Returns the consumer config."
  [& [opts]]
  (merge {:auto.commit.interval.ms 100
          :auto.offset.reset "earliest"
          :bootstrap.servers core/bootstrap-servers
          :enable.auto.commit true
          :group.id (str (java.util.UUID/randomUUID))
          :max.poll.records 1}
         opts))

(defn- make-client
  "Make a `KafkaConsumer` client for `config`."
  [config]
  (KafkaConsumer.
   (core/properties config)
   (or (:key-deserializer config) (deserializer/transit))
   (or (:value-deserializer config) (deserializer/transit))))

(defcomponent Consumer [config]
  :client (make-client config) #(.close %))

(defn consumer
  "Returns a Kafka consumer component."
  [& [opts]]
  (map->Consumer {:config (config opts)}))

(defn list-topics
  "Get metadata about partitions for all topics that the user is
  authorized to view."
  [producer]
  (->> (for [[topic partition-info] (.listTopics (:client producer))]
         [topic (mapv core/partition-info->map partition-info)])
       (into {})))

(defn subscribe!
  [producer topics]
  (.subscribe (:client producer) topics))

(defn poll!
  [producer timeout]
  (map #(vector (.key %) (.value %)) (.poll (:client producer) timeout)))
