(ns burningswell.http.pool
  "A HTTP pool component"
  (:require [clj-http.client :as http]
            [clj-http.conn-mgr :refer [make-reusable-conn-manager]]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [schema.core :as s]))

(def ^:dynamic *defaults*
  "The default clj-http request options."
  {:as :auto
   :coerce :always
   :default-per-route 2
   :insecure? false
   :method :get
   :threads 5
   :throw-exceptions false
   :timeout 5})

(s/defrecord HttpPool
    [as :- s/Keyword
     client :- s/Any
     coerce :- s/Keyword
     connection-manager :- s/Any
     default-per-route :- s/Int
     insecure? :- s/Bool
     method :- (s/enum :get :delete :head :patch :post :put)
     threads :- s/Int
     throw-exceptions :- s/Bool
     timeout :- s/Int]
  {s/Any s/Any})

(s/defn ^:always-validate formatted :- s/Str
  "Returns the formatted config of the HTTP pool."
  [pool :- HttpPool]
  (format (str "%s HTTP pool with %d threads, %ds timeout and "
               "%s connection%s per host")
          (if (:insecure? pool) "Insecure" "Secure")
          (:threads pool)
          (:timeout pool)
          (:default-per-route pool)
          (if (= 1 (:default-per-route pool)) "" "s")))

(s/defn ^:always-validate start-pool :- HttpPool
  "Start the HTTP pool."
  [pool :- HttpPool]
  (if (:connection-manager pool)
    pool
    (let [manager (make-reusable-conn-manager pool)]
      (log/infof "%s started." (formatted pool))
      (assoc pool :connection-manager manager))))

(s/defn ^:always-validate stop-pool :- HttpPool
  "Stop the HTTP pool."
  [pool :- HttpPool]
  (when-let [manager (:connection-manager pool)]
    (.shutdown manager)
    (log/infof "%s stopped." (formatted pool)))
  (assoc pool :connection-manager nil))

(extend-protocol component/Lifecycle
  HttpPool
  (start [pool]
    (start-pool pool))
  (stop [pool]
    (stop-pool pool)))

(s/defn ^:always-validate http-pool :- HttpPool
  "Return a new HTTP pool."
  [& [config]]
  (map->HttpPool (merge *defaults* config)))

(s/defn ^:always-validate request :- s/Any
  "Send a HTTP `request` via `pool`."
  [pool :- HttpPool request]
  ((or (:client pool) #'http/request)
   (merge pool request)))
