(ns burningswell.geoip.maxmind
  (:require [burningswell.geoip.core :as core]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [inflections.core :refer [hyphenate-keys]])
  (:import [com.maxmind.geoip2 DatabaseReader DatabaseReader$Builder]
           [com.maxmind.geoip2.exception AddressNotFoundException]
           java.net.InetAddress
           java.util.zip.GZIPInputStream))

(def root-url
  "The root URL of the Maxmind databases."
  "http://geolite.maxmind.com/download/geoip/database/")

(def url-by-type
  "The URLs of the GeoLite2 database."
  {:city (str root-url "/GeoLite2-City.mmdb.gz")
   :country (str root-url "/GeoLite2-Country.mmdb.gz")})

(def ^:dynamic *defaults*
  {:cache-dir (System/getProperty "java.io.tmpdir")
   :type :city})

(defn strip-gz
  "Strip the gzip extension from `s`."
  [s]
  (str/replace s #".gz$" ""))

(defn cache-file
  "Return the cache path for `url`."
  [db]
  (let [filename (strip-gz (.getName (io/file (:url db))))]
    (io/file (:cache-dir db) filename)))

(defn download-database
  "Download the database from `url` to `file`."
  [url file]
  (let [file (io/file file)]
    (io/make-parents file)
    (with-open [input (GZIPInputStream. (io/input-stream url))
                output (io/output-stream file)]
      (io/copy input output))))

(defn parse-ip
  "Parse an internet address."
  [s]
  (when-not (str/blank? s)
    (try (InetAddress/getByName s)
         (catch Exception _))))

(defn open-database
  "Open a GeoIP database."
  [db]
  (-> (io/input-stream (cache-file db))
      (DatabaseReader$Builder.)
      (.build)))

(defn- decode-map [m]
  (-> (bean m)
      (dissoc :class)
      (hyphenate-keys)))

(defmulti decode
  "Decode the `response`."
  (fn [db response] (:type db)))

(defmethod decode :city [db response]
  (when response
    {:city (decode-map (.getCity response))
     :continent (decode-map (.getContinent response))
     :country (decode-map (.getCountry response))
     :location (decode-map (.getLocation response))
     :postal (decode-map (.getPostal response))}))

(defmethod decode :country [db response]
  (when response
    {:continent (decode-map (.getContinent response))
     :country (decode-map (.getCountry response))}))

(defn started?
  "Returns true if `db` has been started, otherwise false."
  [db]
  (instance? DatabaseReader (:reader db)))

(defn geocode-db
  "Return a new database."
  [{:keys [reader] :as db} ip]
  {:pre [(started? db)]}
  (when-let [ip (parse-ip ip)]
    (try (decode db (case (:type db)
                      :city (.city reader ip)
                      :country (.country reader ip)))
         (catch AddressNotFoundException e
           (log/warnf "Can't geocode ip address: %s" ip)))))

(defn cache-db
  "Cache the database if needed."
  [db]
  (when-not (.exists (cache-file db))
    (download-database (:url db) (cache-file db))
    (log/infof "Maxmind %s database successfully cached." (:type db)))
  db)

(defn start-db
  "Start the GeoIP database."
  [db]
  (if (started? db)
    db
    (do (cache-db db)
        (let [reader (open-database db)]
          (log/info "Maxmind geocoder successfully started.")
          (assoc db :reader reader)))))

(defn stop-db
  "Stop the GeoIP database."
  [db]
  (when (started? db)
    (log/info "Maxmind geocoder successfully stopped."))
  (assoc db :reader nil))

(defrecord Database [reader]
  component/Lifecycle
  (start [db]
    (start-db db))
  (stop [db]
    (stop-db db))
  core/IGeocode
  (geocode [db ip]
    (geocode-db db ip)))

(defn database
  "Return a new database."
  [& [config]]
  (let [type (or (:type config) (:type *defaults*))]
    (-> (merge {:url (url-by-type type)} *defaults* config)
        (map->Database))))

(defmacro with-db
  "Start a new GeoIP database using `config`, bind the started
  database to `db-sym`, evaluate `body` and stop the database again."
  [[db-sym config] & body]
  `(let [db# (component/start (database ~config)), ~db-sym db#]
     (try ~@body (finally (component/stop db#)))))
