(ns burningswell.db.regions
  (:require [burningswell.db.page-views :as page-views]
            [burningswell.db.util :as util]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [datumbazo.record :as record]
            [datumbazo.table :as t]))

(t/deftable regions
  "The regions table."
  (t/column :airport-count :integer)
  (t/column :country-id :integer)
  (t/column :created-at :timestamp :not-null? true)
  (t/column :geom :geography)
  (t/column :id :integer :primary-key? true)
  (t/column :location :geography)
  (t/column :name :citext)
  (t/column :photo-id :integer)
  (t/column :port-count :integer)
  (t/column :spot-count :integer)
  (t/column :updated-at :timestamp :not-null? true)
  (t/column :user-count :integer))

(defn- select-expr [db]
  (record/select-columns db Region))

(defn andalucia
  "Return the region Andalucía, Spain."
  [db]
  (first (by-name db "Andalucía")))

(s/fdef andalucia
  :args (s/cat :db sql/db?)
  :ret (s/nilable ::region))

(defn bali
  "Return the region Bali, Indonesia."
  [db]
  (first (by-name db "Bali")))

(s/fdef bali
  :args (s/cat :db sql/db?)
  :ret (s/nilable ::region))

(defn bizkaia
  "Return the region Bizkaia, Spain."
  [db]
  (first (by-name db "Bizkaia")))

(s/fdef bizkaia
  :args (s/cat :db sql/db?)
  :ret (s/nilable ::region))

(defn hawaii
  "Return the region Hawaii, United States."
  [db]
  (first (by-name db "Hawaii")))

(s/fdef hawaii
  :args (s/cat :db sql/db?)
  :ret (s/nilable ::region))

(defn pais-vasco
  "Return the region País Vasco, Spain."
  [db]
  (first (by-name db "País Vasco")))

(s/fdef pais-vasco
  :args (s/cat :db sql/db?)
  :ret (s/nilable ::region))

(defn as-geo-json
  "Return the `region` as GeoJSON."
  [db region]
  (->> @(util/geo-json-feature-collection
         db (sql/select db [(sql/as "Feature" :type)
                            (sql/as '(cast (st_asgeojson :regions.geom) :json)
                                    :geometry)]
              (sql/from :regions)
              (sql/where `(= :regions.id ~(:id region)))))
       first :feature-collection))

(s/fdef as-geo-json
  :args (s/cat :db sql/db? :region ::region))

(defn as-svg
  "Return the `region` as SVG."
  [db region]
  (->> @(sql/select db [(sql/as '(st_assvg (cast :geom :geometry)) :svg)]
          (sql/from :regions)
          (sql/where `(= :regions.id ~(:id region))))
       first :svg))

(s/fdef as-svg
  :args (s/cat :db sql/db? :region ::region))

(defn as-png
  "Return the `region` as PNG."
  [db region & [opts]]
  (let [{:keys [width height color]} opts]
    (->> @(sql/select db [(sql/as (util/geometry-as-png width height color) :png)]
            (sql/from :regions)
            (sql/where `(= :regions.id ~(:id region))))
         first :png)))

(s/fdef as-png
  :args (s/cat :db sql/db? :region ::region))

(defn within-distance
  "Return the regions in `db` within `distance` in km to `location`."
  [db location distance & [opts]]
  @(sql/select db (select-expr db)
     (sql/from :regions)
     (util/within-distance-to :regions.geom location distance)
     (util/order-by-distance :regions.geom location opts)))

(s/fdef within-distance
  :args (s/cat :db sql/db? :location ::location :distance number? :opts (s/? map?))
  :ret (s/coll-of ::region))

(defn closest
  "Find the closest region to `location`."
  [db location & [distance]]
  (or (first (within-distance db location (or distance 20)))
      (first (by-location db location))))

(s/fdef closest
  :args (s/cat :db sql/db? :location ::location :distance (s/? number?))
  :ret (s/coll-of ::region))

(defn by-spot
  "Returns the region of `spot` from `db`."
  [db spot]
  (some->> spot :region-id (by-id db)))

(s/fdef by-spot
  :args (s/cat :db sql/db? :spot map?)
  :ret (s/nilable ::region))

(defn page-views
  "Returns the page views for `region`."
  [db region & [opts]]
  (first (page-views/totals db :regions [region] opts)))

(s/fdef page-views
  :args (s/cat :db sql/db? :region ::region :opts (s/? (s/nilable map?))))

(defn search
  "Search regions."
  [db & [{:keys [countries direction distance except limit location
                 min-spots offset sort query] :as opts}]]
  @(sql/select db [:regions.id]
     (sql/from :search.regions)
     (util/fulltext query :name)
     (util/within-distance-to :geom location distance {:spheroid? true})
     (when min-spots
       (sql/where `(>= :spot-count ~min-spots) :and))
     (when-not (empty? countries)
       (sql/where `(in :country-id ~(seq countries)) :and))
     (when-not (empty? except)
       (sql/where `(not (in :id ~(seq except))) :and))
     (sql/limit limit)
     (sql/offset offset)
     (cond
       (and location (= sort :distance))
       (util/order-by-distance :location location opts)
       (= sort :views)
       (util/order-by-views :regions opts)
       sort
       (util/order-by :regions opts)
       :else (sql/order-by :name))))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? map?))
  :ret (s/coll-of ::region))

(defn select-top-photos
  "Return the top photo for each region."
  [db]
  (sql/select db (sql/distinct
                  [:photos-regions.region-id
                   :photos-regions.photo-id]
                  :on [:photos-regions.region-id] )
    (sql/from :photos-regions)
    (sql/join :photos.id :photos-regions.photo-id)
    (sql/order-by :photos-regions.region-id
                  (sql/desc :photos.created-at))))

(s/fdef select-top-photos
  :args (s/cat :db sql/db?))

(defn update-photos!
  "Update the photos of all regions."
  [db]
  (first @(sql/update db :regions
            {:photo-id :top-photos.photo-id}
            (sql/from (sql/as (select-top-photos db) :top-photos))
            (sql/where `(= :regions.id :top-photos.region-id)))))

(s/fdef update-photos!
  :args (s/cat :db sql/db?))

;; (defn- order-by-regions
;;   "Add an ORDER BY clause to region queries."
;;   [{:keys [bounding-box location] :as opts}]
;;   (cond
;;     bounding-box
;;     (order-by-bounding-box :regions.location bounding-box)
;;     location
;;     (order-by-distance :regions.location location)
;;     :else
;;     (order-by :regions.name)))

;; (defn- select-all [db & [opts]]
;;   (let [{:keys [bounding-box distance location min-spots]} opts]
;;     (select db [:regions.id
;;                 :regions.country-id
;;                 :regions.photo-id
;;                 :regions.name
;;                 :regions.geom
;;                 (as '(cast :regions.location :geometry) :location)
;;                 :regions.airport-count
;;                 :regions.port-count
;;                 :regions.spot-count
;;                 :regions.user-count
;;                 :regions.created-at
;;                 :regions.updated-at]
;;       (from :regions)
;;       (fulltext (:query opts) :regions.name)
;;       (within-bounding-box :regions.geom bounding-box)
;;       (within-distance-to :regions.geom location distance)
;;       (when min-spots
;;         (where `(>= :regions.spot-count (cast ~min-spots :integer)) :and))
;;       (paginate (:page opts) (:per-page opts))
;;       (order-by-regions opts))))

;; (defn- select-details [db & [opts]]
;;   (let [{:keys [bounding-box distance location min-spots]} opts]
;;     (select db [:regions.id
;;                 :regions.name
;;                 :regions.location
;;                 :regions.airport-count
;;                 :regions.port-count
;;                 :regions.spot-count
;;                 :regions.user-count
;;                 :regions.created-at
;;                 :regions.updated-at
;;                 (as `(json_build_object
;;                       "country" (json-embed-country :countries)
;;                       "photo" ~embedded-photo)
;;                     :_embedded)]
;;       (from (as (select-all db opts) :regions))
;;       (join :countries.id :regions.country-id)
;;       (join :photos.id :regions.photo-id :type :left)
;;       (order-by-regions opts))))

;; (defn exists?
;;   "Returns true if `region` exists in `db`, otherwise false."
;;   [db region]
;;   (not (empty? @(sql/select db [:id]
;;                   (sql/from :regions)
;;                   (sql/where `(= :id ~(:id region)))))))

;; (s/defn by-country :- [Region]
;;   "Return the regions of `country` in `db`."
;;   [db :- Database country :- Country & [opts]]
;;   @(compose
;;     (select-details db (dissoc-pagination opts))
;;     (where `(= :regions.country-id ~(:id country)))
;;     (paginate (:page opts) (:per-page opts))))

;; (s/defn by-id :- (s/maybe Region)
;;   "Return the region in `db` by `id`."
;;   [db :- Database id :- s/Num]
;;   (first @(compose
;;            (select-details db)
;;            (where `(= :regions.id (cast ~id :integer))))))

;; (s/defn by-ids :- [Region]
;;   "Return all regions in `db` by the list of `ids`."
;;   [db :- Database ids :- [s/Num] & [opts]]
;;   @(compose
;;     (select-details db (dissoc-pagination opts))
;;     (where `(in :regions.id ~ids))
;;     (paginate (:page opts) (:per-page opts))))

;; (s/defn by-name :- (s/maybe Region)
;;   "Return the country in `db` by `name`."
;;   [db :- Database name :- s/Str]
;;   (first @(compose
;;            (select-details db)
;;            (where `(= :regions.name ~name)))))

;; (defn by-location
;;   "Return the country in `db` by `location`."
;;   [db location]
;;   (let [location (geo/geometry location)]
;;     @(sql/select db [:*]
;;        (sql/from :regions)
;;        (sql/where `(st_intersects :regions.geom (cast ~location :geography)))
;;        (sql/order-by `(st_distance (cast ~location :geography) :regions.geom)))))

;; (defn- row [continent]
;;   (assoc (select-keys continent [:name :location])
;;          :country-id (-> continent :_embedded :country :id)))

;; (s/defn delete
;;   "Delete `region` from `db`."
;;   [db :- Database region :- Region]
;;   (->> @(sql/delete db :regions
;;           (where `(= :regions.id
;;                      ~(:id region))))
;;        first :count))

;; (s/defn in-continent :- [Region]
;;   "Return all regions in `continent`."
;;   [db :- Database continent :- Continent & [opts]]
;;   @(compose
;;     (select-details db (dissoc-pagination opts))
;;     (where `(= :countries.continent-id ~(:id continent)))
;;     (paginate (:page opts) (:per-page opts))))

;; (s/defn in-country :- [Region]
;;   "Return all regions in `country`."
;;   [db :- Database country :- Country & [opts]]
;;   @(compose
;;     (select-details db (dissoc-pagination opts))
;;     (where `(= :regions.country-id ~(:id country)) :and)
;;     (paginate (:page opts) (:per-page opts))))

(defn update-counters!
  "Update the counters of all regions."
  [db]
  (->> [(sql/select db ['(update-region-airport-count)])
        (sql/select db ['(update-region-port-count)])
        (sql/select db ['(update-region-spot-count)])
        (sql/select db ['(update-region-user-count)])]
       (map (comp first deref))
       (apply merge)))

;; (s/defn assoc-photo :- [Region]
;;   "Assoc the photo and their images to all `regions`."
;;   [db :- Database regions :- [Region]]
;;   (let [photos (zip-by-id (photos/by-regions db regions {:images true}))]
;;     (map #(->> (get photos (-> % :_embedded :photo :id))
;;                (assoc-in % [:_embedded :photo]))
;;          regions)))

;; (s/defn geometry :- GeoJSON
;;   "Return the geometry of `region` as GeoJSON data structure."
;;   [db :- Database region :- Region]
;;   (geojson-by-column db :regions :geom :id (:id region)))

;; (defn nearby-region
;;   "Returns all regions nearby `region`."
;;   [db region & [opts]]
;;   @(compose
;;     (select-details
;;      db (-> (dissoc-pagination opts)
;;             (assoc :location (:location region))))
;;     (where `(<> :regions.id ~(:id region)) :and)
;;     (paginate (:page opts) (:per-page opts))))

(comment

  (def my-db (-> reloaded.repl/system))

  (burningswell.db.continents/update-photos! my-db)
  (burningswell.db.countries/update-photos! my-db)
  (burningswell.db.regions/update-photos! my-db)

  )
