(ns burningswell.db.util
  (:require [burningswell.json :as json]
            [datumbazo.core :as sql]
            [geo.postgis :refer [geometry IGeometry point]]
            [no.en.core :refer [parse-double]]
            [clojure.string :as str])
  (:import java.awt.Color))

(defn color-vector
  "Convert a java.awt.Color into a vector."
  [color]
  [(.getRed color)
   (.getGreen color)
   (.getBlue color)])

(defn geometry-as-png [width height color]
  `(st_aspng
    (st_asraster
     (cast :geom :geometry)
     (cast ~(or width 400) :integer)
     (cast ~(or height 400) :integer)
     ["8BUI" "8BUI" "8BUI"]
     ;; TODO; Is this really RGB?
     ~(color-vector (or color (Color. 0 0 1)))
     ~(color-vector (Color. 0 0 0)))))

(def embedded-photo
  '(case (is-null :photos.id) nil
         (json_build_object
          "id" :photos.id
          "title" :photos.title
          "flickr"
          (json_build_object
           "id" :photos.flickr-id
           "owner"
           (json_build_object
            "id" :photos.flickr-owner-id
            "name" :photos.flickr-owner-name
            "url" :photos.flickr-owner-url)))))

(defn embedded-user-settings [table]
  `(case (is-null ~(keyword (str (name table) ".id"))) nil
         (json_build_object
          "units" ~(keyword (str (name table) ".units")))))

(defn geo-json-feature-collection
  "Select a GeoJSON feature collection using `feature-query`. "
  [db feature-query]
  (sql/select db [(sql/as '(cast (row_to_json :feature-collections) :text)
                          :feature-collection)]
              (sql/from (sql/as (sql/select db [(sql/as "FeatureCollection" :type)
                                                (sql/as '(array_to_json (array_agg :feature-query))
                                                        :features)]
                                            (sql/from (sql/as feature-query :feature-query)))
                                :feature-collections))))

;; TODO: Whey is geo.core/parse-location not working?
(defn parse-location [s]
  (if (string? s)
    (let [parts (->> (str/split s #"\s*,\s*")
                     (map parse-double)
                     (remove nil?))]
      (if (= 2 (count parts))
        (apply point 4326 (reverse parts))))
    s))

(defn fulltext
  "Add a where condition to a select query."
  [query & columns]
  (when-not (str/blank? query)
    (sql/where `(or (~(keyword "@@")
                     (to_tsvector (array_to_string [~@columns], " "))
                     (to_tsquery ~query))
                    ~@(map #(list 'like %1 (str "%" query "%")) columns))
               :and)))

(defn order-by-distance
  [geom-1 geom-2 & [{:keys [direction]}]]
  (let [geom-1 (parse-location geom-1)
        geom-2 (parse-location geom-2)]
    (when (and geom-1 geom-2)
      (sql/order-by
       ((if (= direction :desc) sql/desc sql/asc)
        `(<-> (st_centroid (cast ~(geometry geom-1) :geometry))
              (st_centroid (cast ~(geometry geom-2) :geometry))))))))

(defn order-by-bounding-box
  "Restrict the result set to rows that have `column` within
  `bounding-box`."
  [column bounding-box & [opts]]
  (when bounding-box
    (sql/order-by
     `(st_distance
       ~column
       (st_centroid
        (st_transform
         (st_makeenvelope
          ~(.getX (.getLLB bounding-box))
          ~(.getY (.getLLB bounding-box))
          ~(.getX (.getURT bounding-box))
          ~(.getY (.getURT bounding-box))
          ~(.getSrid (.getURT bounding-box)))
         (st_srid ~column)))))))

(defn order-by
  "Returns a SQL order by clause for `opts`."
  [table & [{:keys [direction nulls sort] :as opts}]]
  (when sort
    (let [direction (if (= direction :desc) sql/desc sql/asc)
          nulls (or nulls :last)]
      (sql/order-by (sql/nulls (direction (keyword sort)) nulls)))))

(defn order-by-views
  "Returns a order by page views SQL clause for `opts`."
  [table & [{:keys [direction nulls sort] :as opts}]]
  (when (= sort :views)
    (->> {:direction (if (= direction :desc) :asc :desc)
          :sort (keyword (str "page-views." (name table) ".total"))
          :nulls (or nulls :last)}
         (merge opts)
         (order-by table))))

(defn within-bounding-box
  "Restrict the result set to rows that have `column` within
  `bounding-box`."
  [column bounding-box & [opts]]
  (when bounding-box
    (let [srid (or (:srid opts) 4326)]
      (sql/compose
       (sql/where `(:&&
                    ~column
                    (cast (st_transform
                           (st_makeenvelope
                            ~(.getX (.getLLB bounding-box))
                            ~(.getY (.getLLB bounding-box))
                            ~(.getX (.getURT bounding-box))
                            ~(.getY (.getURT bounding-box))
                            ~(.getSrid (.getURT bounding-box)))
                           (st_srid ~column))
                          :geography))
                  :and)
       (order-by-bounding-box column bounding-box)))))

(defn within-distance-to
  "Return a WHERE clause where `column` is within `distance`
  kilometers to `location`."
  [column location distance & [{:keys [spheroid?]}]]
  (when (and location distance)
    (sql/where
     `(st_dwithin
       (cast ~(geometry location) :geography)
       (cast ~column :geography)
       (* (cast ~(or distance 1000) :double-precision) 1000)
       ~(true? spheroid?))
     :and)))

(defn assoc-embedded [m & kv]
  (reduce
   (fn [m [k v]]
     (if-not v
       m (assoc-in m [:_embedded k] v)))
   m (partition 2 kv)))

(extend-type clojure.lang.Keyword
  IGeometry
  (geometry [k]
    k))

(defn zip-by-key
  "Zip the elements of `coll` by their `key`."
  [coll & ks]
  (zipmap (map #(get-in % ks) coll) coll))

(defn zip-by-id
  "Zip the elements of `coll` by their :id key."
  [coll]
  (zip-by-key coll :id))

(defn enabled-gdal-drivers
  "Return the short and long name of the enabled GDAL drivers."
  [db]
  (->> @(sql/select db [(sql/as :short_name :short-name)
                        (sql/as :long_name :long-name)]
                    (sql/from (sql/as '(st_gdaldrivers) :drivers))
                    (order-by 1))
       (identity)))

(defn geojson-by-column
  "Return the GeoJSON of a row by a column."
  [db table geo-column id-column id-value]
  (assoc (->> @(sql/select db [(sql/as `(st_asgeojson ~geo-column) :json)]
                           (sql/from table)
                           (sql/where `(= ~id-column ~id-value)))
              first :json json/read-json)
         id-column id-value))

(defn fetch-records-by-ids [db f opts records]
  (f db (map :id records) (dissoc opts :page :per-page)))

(defn select-count-rows
  "Return a select statement that counts all rows in `table`."
  [db table]
  (sql/select db ['(count :*)]
              (sql/from table)))

(defn random-row
  "Return a random row from `table` in `db`."
  [db table]
  (first @(sql/select db [:*]
                      (sql/from table)
                      (sql/limit 1)
                      (sql/offset `(floor (* (random) ~(select-count-rows db table)))))))

(defn dissoc-pagination
  "Remove the pagination keys :page and :per-page from `m`."
  [m]
  (dissoc m :page :per-page))
