(ns burningswell.db.search
  (:require [burningswell.db.util :refer :all]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [inflections.core :as infl]
            [clojure.string :as str]))

(defn- rank
  "Returns the search rank expression."
  [query]
  (sql/as `(cast (round (cast (word_similarity :name ~query) :numeric) 3) :double-precision) :rank))

(defn- result-type
  "Returns the result type expression."
  [table]
  (sql/as (name (infl/singular table)) :type))

(defn search-clause [table columns query]
  (sql/where
   `(or ~@(for [column columns
                term (str/split query #"\s+")]
            `(ilike ~(keyword (str (name table) "." (name column)))
                    ~(str "%" term "%"))))
   :and))

(defn search
  "Return all comments in `db`."
  [db & [{:keys [limit offset query min-spots] :as opts}]]
  (when-not (str/blank? query)
    (->> @(sql/union
           (sql/select db [:id
                           (sql/as :name :term)
                           (result-type :continent)
                           (rank query)]
             (sql/from :continents)
             (search-clause :continents [:name] query))
           (sql/select db [:id
                           (sql/as :name :term)
                           (result-type :country)
                           (rank query)]
             (sql/from :countries)
             (when min-spots (sql/where `(> :spot-count ~min-spots)))
             (search-clause :countries [:name] query))
           (sql/select db [:id
                           (sql/as :name :term)
                           (result-type :region)
                           (rank query)]
             (sql/from :regions)
             (when min-spots (sql/where `(> :spot-count ~min-spots)))
             (search-clause :regions [:name] query))
           (sql/select db [:id
                           (sql/as :name :term)
                           (result-type :spot)
                           (rank query)]
             (sql/from :spots)
             (search-clause :spots [:name] query)
             (sql/offset offset)
             (sql/limit limit)
             (sql/order-by (sql/desc :rank))))
         (map #(update-in % [:type] keyword)))))

(s/def :burningswell.db.search.opts/limit (s/nilable nat-int?))
(s/def :burningswell.db.search.opts/offset (s/nilable nat-int?))
(s/def :burningswell.db.search.opts/query (s/nilable string?))

(s/def ::search-opts
  (s/keys :opt-un [:burningswell.db.search.opts/limit
                   :burningswell.db.search.opts/offset
                   :burningswell.db.search.opts/query]))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? (s/nilable ::search-opts))))
