(ns burningswell.db.spots
  (:require [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [datumbazo.table :as t]
            [burningswell.db.util :as util]))

(t/deftable spots
  "The spots table."
  (t/column :country-id :integer)
  (t/column :created-at :timestamp :not-null? true)
  (t/column :id :integer :primary-key? true)
  (t/column :location :geography :geometry :point :srid 4326)
  (t/column :name :citext)
  (t/column :photo-id :integer)
  (t/column :region-id :integer)
  (t/column :time-zone-id :integer)
  (t/column :updated-at :timestamp :not-null? true)
  (t/column :user-id :integer)
  (t/column :visible :text))

(defn banzai-pipeline
  "Returns the spot Banzai Pipline, Hawaii, United States."
  [db]
  (first (by-name db "Banzai Pipeline")))

(s/fdef banzai-pipeline
  :args (s/cat :db sql/db?))

(defn menakoz
  "Returns the spot Meñakoz, Spain."
  [db]
  (first (by-name db "Meñakoz")))

(s/fdef menakoz
  :args (s/cat :db sql/db?))

(defn mundaka
  "Returns the spot Mundaka, Spain."
  [db]
  (first (by-name db "Mundaka")))

(s/fdef mundaka
  :args (s/cat :db sql/db?))

(defn padang-padang
  "Returns the spot Padang Padang, Indonesia."
  [db]
  (first (by-name db "Padang Padang")))

(s/fdef padang-padang
  :args (s/cat :db sql/db?))

(defn uluwatu
  "Returns the spot Uluwatu, Indonesia."
  [db]
  (first (by-name db "Uluwatu")))

(s/fdef uluwatu
  :args (s/cat :db sql/db?))

(defn sao-lorenco
  "Returns the spot São Lorenço, Portugal."
  [db]
  (first (by-name db "São Lorenço")))

(s/fdef sao-lorenco
  :args (s/cat :db sql/db?))

(defn add-photo!
  [db spot photo & [user]]
  @(sql/insert db :photos-spots []
     (sql/values [{:photo-id (:id photo)
                   :spot-id (:id spot)
                   :user-id (:id user)}])
     (sql/on-conflict [:photo-id :spot-id]
       (sql/do-nothing))
     (sql/returning :*)))

(s/fdef add-photo!
  :args (s/cat :db sql/db? :spot map? :photo map? :user (s/? map?)))

(defn delete-photos!
  "Delete all the photos of `spot` in `db`."
  [db spot]
  @(sql/delete db :photos
     (sql/where `(in :id ~(sql/select db [:photo-id]
                            (sql/from :photos-spots)
                            (sql/where `(= :photos-spots.spot-id ~(:id spot))))))))

(s/fdef delete-photos!
  :args (s/cat :db sql/db? :spot map?))

(defn save-cover-photo!
  "Save `photo` as the cover photo of `spot`."
  [db spot photo]
  @(sql/update db :spots {:photo-id (:id photo)}
               (sql/where `(= :id ~(:id spot)))))

(s/fdef save-cover-photo!
  :args (s/cat :db sql/db? :spot map? :photo map?))

(defn search
  "Search spots."
  [db & [{:keys [countries direction distance except limit location
                 offset regions sort users query] :as opts}]]
  @(sql/select db [:spots.id]
     (sql/from :spots)
     (util/fulltext query :spots.name)
     (util/within-distance-to :location location distance {:spheroid? true})
     (when-not (empty? countries)
       (sql/where `(in :country-id ~(seq countries)) :and))
     (when-not (empty? except)
       (sql/where `(not (in :id ~(seq except))) :and))
     (when-not (empty? regions)
       (sql/where `(in :region-id ~(seq regions)) :and))
     (when-not (empty? users)
       (sql/where `(in :user-id ~(seq users)) :and))
     (some-> limit sql/limit)
     (some-> offset sql/offset)
     (cond
       (and location (= sort :distance))
       (util/order-by-distance :location location opts)
       sort
       (util/order-by :spots opts)
       :else (sql/order-by :spots.name))))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? map?))
  :ret (s/coll-of ::spot))

(defn set-time-zones!
  "Set the timezone of all spots."
  [db]
  (->> @(sql/update db :spots
          {:time-zone-id :t.id}
          (sql/from (sql/as (sql/select db [:time-zones.id (sql/as :spots.id :spot-id)]
                              (sql/from :time-zones)
                              (sql/join :spots
                                        '(on (st_intersects
                                              :spots.location
                                              :time-zones.geom))))
                            :t))
          (sql/where '(= :spots.id :t.spot-id)))
       first :count))

(s/fdef set-time-zones!
  :args (s/cat :db sql/db? :opts (s/? map?))
  :ret nat-int?)

;; (defn- order-by-spots
;;   "Add an ORDER BY clause to spot queries."
;;   [{:keys [bounding-box location] :as opts}]
;;   (cond
;;     bounding-box
;;     (order-by-bounding-box :spots.location bounding-box)
;;     location
;;     (order-by-distance :spots.location location)
;;     :else (order-by :spots.name)))

;; (defn- select-all
;;   "Select all spots."
;;   [db & [{:keys [bounding-box distance location photos] :as opts}]]
;;   (select db [:spots.id
;;               :spots.country-id
;;               :spots.region-id
;;               :spots.user-id
;;               :spots.photo-id
;;               :spots.time-zone-id
;;               :spots.name
;;               (as '(cast :spots.location :geometry) :location)
;;               :spots.visible
;;               :spots.created-at
;;               :spots.updated-at]
;;     (from :spots)
;;     (fulltext (:query opts) :spots.name)
;;     (within-bounding-box :spots.location bounding-box)
;;     (within-distance-to :spots.location location distance)
;;     (when (= photos true)
;;       (where '(is-not-null :spots.photo-id) :and))
;;     (when (= photos false)
;;       (where '(is-null :spots.photo-id) :and))
;;     (paginate (:page opts) (:per-page opts))
;;     (order-by-spots opts)))

;; (defn- select-details
;;   "Select all spots."
;;   [db & [{:keys [distance location photos] :as opts}]]
;;   (select db [:spots.id
;;               :spots.name
;;               :spots.location
;;               :spots.visible
;;               :spots.created-at
;;               :spots.updated-at
;;               (as `(json_build_object
;;                     "country" (json-embed-country :countries)
;;                     "region" (json-embed-region :regions)
;;                     "user" (json-embed-user :users)
;;                     "photo" ~embedded-photo
;;                     "time-zone" (json-embed-time-zone :time-zones))
;;                   :_embedded)]
;;     (from (as (select-all db opts) :spots))
;;     (join :countries.id :spots.country-id :type :left)
;;     (join :regions.id :spots.region-id :type :left)
;;     (join :users.id :spots.user-id :type :left)
;;     (join :photos.id :spots.photo-id :type :left)
;;     (join :time-zones.id :spots.time-zone-id :type :left)
;;     (order-by-spots opts)))

;; (s/defn country-id :- (s/maybe s/Int)
;;   "Return the country id of `spot`."
;;   [spot]
;;   (-> spot :_embedded :country :id))

;; (s/defn region-id :- (s/maybe s/Int)
;;   "Return the region id of `spot`."
;;   [spot]
;;   (-> spot :_embedded :region :id))

;; (s/defn user-id :- (s/maybe s/Int)
;;   "Return the user id of `spot`."
;;   [spot]
;;   (-> spot :_embedded :user :id))

;; (s/defn photo-id :- (s/maybe s/Int)
;;   "Return the photo id of `spot`."
;;   [spot]
;;   (-> spot :_embedded :photo :id))

;; (s/defn by-id
;;   "Return the country in `db` by `id`."
;;   [db :- Database id :- s/Num]
;;   (first @(compose
;;            (select-details db)
;;            (where `(= :spots.id (cast ~id :integer))))))

;; (s/defn by-ids :- [Spot]
;;   "Return all spots in `db` by the list of `ids`."
;;   [db :- Database ids :- [s/Num] & [opts]]
;;   (when (not-empty ids)
;;     @(compose
;;       (select-details db (dissoc-pagination opts))
;;       (where `(in :spots.id ~(seq ids)))
;;       (paginate (:page opts) (:per-page opts)))))

;; (s/defn all :- [Spot]
;;   "Return all spots in `db`."
;;   [db :- Database & [opts]]
;;   @(select-details db opts))

;; (defn select-nearby-location
;;   "Select all spots nearby `location`."
;;   [db location & [opts]]
;;   (select-details db (assoc opts :location location)))

;; (defn nearby-spots
;;   "Returns all spots nearby `spot`."
;;   [db spot & [opts]]
;;   @(compose
;;     (select-nearby-location db (:location spot) opts)
;;     (where `(<> :spots.id ~(:id spot)) :and)))

;; (s/defn by-name
;;   "Return the country in `db` by `name`."
;;   [db :- Database name :- s/Str]
;;   (first @(compose
;;            (select-details db)
;;            (where `(= :spots.name ~name)))))

;; (s/defn by-location :- (s/maybe Spot)
;;   "Return the spot in `db` at `location`."
;;   [db :- Database location :- Point]
;;   (first @(compose
;;            (select-details db)
;;            (where `(= :spots.location ~(geometry location))))))

;; (s/defn within-distance :- [Spot]
;;   "Return the spots in `db` within :distance in km to `location`."
;;   [db :- Database location :- Point distance :- s/Num & [opts]]
;;   @(compose
;;     (select-details db (assoc opts :location location))
;;     (within-distance-to :spots.location location distance)))

;; (defn- row [spot]
;;   (assoc (select-keys spot [:name :location :visible])
;;          :country-id (country-id spot)
;;          :region-id (region-id spot)
;;          :user-id (user-id spot)
;;          :photo-id (photo-id spot)))

;; (s/defn delete
;;   "Delete `spot` from `db`."
;;   [db :- Database spot :- Spot]
;;   (->> @(sql/delete db :spots
;;           (where `(= :spots.id
;;                      ~(:id spot))))
;;        first :count))

;; (s/defn in-continent :- [Spot]
;;   "Returns all spots in `continent`."
;;   [db :- Database continent :- Continent & [opts]]
;;   @(compose
;;     (select-details db (dissoc-pagination opts))
;;     (where `(= :countries.continent-id ~(:id continent)))
;;     (paginate (:page opts) (:per-page opts))))

;; (s/defn in-country :- [Spot]
;;   "Returns all spots in `country`."
;;   [db :- Database country :- Country & [opts]]
;;   @(compose
;;     (select-details db (dissoc-pagination opts))
;;     (where `(= :spots.country-id ~(:id country)))
;;     (paginate (:page opts) (:per-page opts))))

;; (s/defn in-region :- [Spot]
;;   "Returns all spots in `region`."
;;   [db :- Database region :- Region & [opts]]
;;   @(compose
;;     (select-details db (dissoc-pagination opts))
;;     (where `(= :spots.region-id ~(:id region)))
;;     (paginate (:page opts) (:per-page opts))))

;; (s/defn insert
;;   "Insert `spot` into `db`."
;;   [db :- Database spot]
;;   (->> @(sql/insert db :spots []
;;           (values [(row spot)])
;;           (returning :id))
;;        first :id (by-id db)))

;; (s/defn update
;;   "Update `spot` in `db`."
;;   [db :- Database spot]
;;   (->> @(sql/update db :spots (row spot)
;;                     (where `(= :spots.id
;;                                (cast ~(:id spot) :integer)))
;;                     (returning :id))
;;        first :id (by-id db)))

;; (s/defn save :- Spot
;;   "Insert `spot` into `db`."
;;   [db :- Database spot]
;;   (or (update db spot)
;;       (insert db spot)))

;; (s/defn assoc-photo :- [Spot]
;;   "Assoc the photo and their images to all `spots`."
;;   [db :- Database spots :- [Spot]]
;;   (let [photos (zip-by-id (photos/by-spots db spots {:images true}))]
;;     (map #(->> (get photos (-> % :_embedded :photo :id))
;;                (assoc-in % [:_embedded :photo]))
;;          spots)))

;; (s/defn assoc-current-weather :- [Spot]
;;   "Assoc the current weather forecasts to `spots`."
;;   [db :- Database spots :- [Spot] & [opts]]
;;   (let [weather (weather/current-weather-by-spots db spots opts)]
;;     (map #(assoc-in % [:_embedded :weather]
;;                     (get weather (:id %)))
;;          spots)))

;; (s/defn without-a-photo :- [Spot]
;;   "Return all spots that don't have a photo."
;;   [db :- Database & [opts]]
;;   @(compose
;;     (select-details db (dissoc-pagination opts))
;;     (where `(is-null :spots.photo-id))
;;     (paginate (:page opts) (:per-page opts))))

;; (s/defn set-timezone
;;   "Set the timezone of all spots."
;;   [db :- Database & [opts]]
;;   (->> @(sql/update db :spots
;;           {:time-zone-id :t.id}
;;           (from (as (select db [:time-zones.id (as :spots.id :spot-id)]
;;                       (from :time-zones)
;;                       (join :spots
;;                             '(on (st_intersects
;;                                   :spots.location
;;                                   :time-zones.geom))))
;;                     :t))
;;           (where '(= :spots.id :t.spot-id)))
;;        first :count))

;; (defn location-available?
;;   "Returns true if there is no other spot within a distance of 100m or
;;   `distance` around `location`, otherwise false."
;;   [db location & [opts]]
;;   (let [distance (or (:distance opts) 0.1)]
;;     (empty? @(sql/select db [:id]
;;                (sql/from :spots)
;;                (within-distance-to
;;                 :location location distance {:spheroid? true})
;;                (when-let [ids (not-empty (:exclude opts))]
;;                  (sql/where `(not (in :id ~(seq ids))) :and))
;;                (sql/limit 1)))))

;; (defn exists?
;;   "Returns true if `country` exists in `db`, otherwise false."
;;   [db spot]
;;   (not (empty? @(sql/select db [:id]
;;                   (sql/from :spots)
;;                   (sql/where `(= :id ~(:id spot)))))))


;; (defn save-cover-photo
;;   "Save `photo` as the cover photo of `spot`."
;;   [db spot photo]
;;   @(sql/update db :spots {:photo-id (:id photo)}
;;                (sql/where `(= :id ~(:id spot)))))
