(ns burningswell.db.search
  (:require [burningswell.db.util :refer :all]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [inflections.core :as infl]
            [clojure.string :as str]))

(defn refresh-views
  "Refresh the search views."
  [db]
  @(sql/refresh-materialized-view db :search.autocomplete
     (sql/concurrently true)))

(s/fdef refresh-views
  :args (s/cat :db sql/db?))

(defn- rank
  "Returns the search rank expression."
  [query]
  (sql/as `(cast
            (ts_rank_cd
             (to_tsvector :name)
             (to_tsquery ~query))
            :decimal)
          :rank))

(defn- result-type
  "Returns the result type expression."
  [table]
  (sql/as (name (infl/singular table)) :type))

(defn search
  "Return all comments in `db`."
  [db & [{:keys [limit offset query min-spots] :as opts}]]
  (when-not (str/blank? query)
    (->> @(sql/union
           (sql/select db [:id
                           (sql/as :name :term)
                           (result-type :continent)
                           (rank query)]
             (sql/from :continents)
             (fulltext query :name))
           (sql/select db [:id
                           (sql/as :name :term)
                           (result-type :country)
                           (rank query)]
             (sql/from :countries)
             (when min-spots (sql/where `(> :spot-count ~min-spots)))
             (fulltext query :name))
           (sql/select db [:id
                           (sql/as :name :term)
                           (result-type :region)
                           (rank query)]
             (sql/from :regions)
             (when min-spots (sql/where `(> :spot-count ~min-spots)))
             (fulltext query :name))
           (sql/select db [:id
                           (sql/as :name :term)
                           (result-type :spot)
                           (rank query)]
             (sql/from :spots)
             (fulltext query :name)
             (sql/offset offset)
             (sql/limit limit)
             (sql/order-by :rank :term)))
         (map #(update-in % [:type] keyword)))))

(s/def :burningswell.db.search.opts/limit (s/nilable nat-int?))
(s/def :burningswell.db.search.opts/offset (s/nilable nat-int?))
(s/def :burningswell.db.search.opts/query (s/nilable string?))

(s/def ::search-opts
  (s/keys :opt-un [:burningswell.db.search.opts/limit
                   :burningswell.db.search.opts/offset
                   :burningswell.db.search.opts/query]))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? (s/nilable ::search-opts))))
