(ns burningswell.db.roles
  (:require [burningswell.db.util :as util]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [datumbazo.table :as t]))

(t/deftable roles
  "The roles table."
  (t/column :created-at :timestamp :not-null? true)
  (t/column :description :citext)
  (t/column :id :integer :primary-key? true)
  (t/column :name :citext :not-null? true :unique? true)
  (t/column :updated-at :timestamp :not-null? true))

(defn admin
  "Returns the admin role."
  [db]
  (by-id db 1))

(s/fdef admin
  :args (s/cat :db sql/db?)
  :ret ::role)

(defn by-user
  "Find all roles of `user` in `db`."
  [db user & [opts]]
  @(sql/select db [:roles.*]
     (sql/from :roles)
     (sql/join :roles-users.role-id :roles.id)
     (sql/where `(= :roles-users.user-id ~(:id user)))))

(defn editor
  "Returns the editor role."
  [db]
  (by-id db 2))

(s/fdef editor
  :args (s/cat :db sql/db?)
  :ret ::role)

(defn surfer
  "Returns the surfer role."
  [db]
  (by-id db 3))

(s/fdef surfer
  :args (s/cat :db sql/db?)
  :ret ::role)

(defn search
  "Search roles."
  [db & [{:keys [direction limit offset sort query] :as opts}]]
  @(sql/select db [:roles.id]
     (sql/from :roles)
     (util/fulltext query :name :description)
     (sql/limit limit)
     (sql/offset offset)
     (util/order-by :roles opts)))

(s/fdef search
  :args (s/cat :db sql/db? :opts (s/? map?))
  :ret (s/coll-of ::role))
