(ns burningswell.db.util
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.json :as json]
            [clojure.set :refer [rename-keys]]
            [clojure.string :refer [blank? split]]
            [datumbazo.core :refer :all]
            [no.en.core :refer [parse-double]]
            [geo.core :as geo]
            [geo.postgis :refer [IGeometry geometry point]]
            [datumbazo.core :as sql])
  (:import java.awt.Color
           geo.core.IPoint))

(defn color-vector
  "Convert a java.awt.Color into a vector."
  [color]
  [(.getRed color)
   (.getGreen color)
   (.getBlue color)])

(defn geometry-as-png [width height color]
  `(st_aspng
    (st_asraster
     (cast :geom :geometry)
     (cast ~(or width 400) :integer)
     (cast ~(or height 400) :integer)
     ["8BUI" "8BUI" "8BUI"]
     ;; TODO; Is this really RGB?
     ~(color-vector (or color (Color. 0 0 1)))
     ~(color-vector (Color. 0 0 0)))))

(def embedded-photo
  '(case (is-null :photos.id) nil
         (json_build_object
          "id" :photos.id
          "title" :photos.title
          "flickr"
          (json_build_object
           "id" :photos.flickr-id
           "owner"
           (json_build_object
            "id" :photos.flickr-owner-id
            "name" :photos.flickr-owner-name
            "url" :photos.flickr-owner-url)))))

(defn embedded-user-settings [table]
  `(case (is-null ~(keyword (str (name table) ".id"))) nil
         (json_build_object
          "units" ~(keyword (str (name table) ".units")))))

(defn geo-json-feature-collection
  "Select a GeoJSON feature collection using `feature-query`. "
  [db feature-query]
  (select db [(as '(cast (row_to_json :feature-collections) :text)
                  :feature-collection)]
    (from (as (select db [(as "FeatureCollection" :type)
                          (as '(array_to_json (array_agg :feature-query))
                              :features)]
                (from (as feature-query :feature-query)))
              :feature-collections))))

;; TODO: Whey is geo.core/parse-location not working?
(defn parse-location [s]
  (if (string? s)
    (let [parts (->> (split s #"\s*,\s*")
                     (map parse-double)
                     (remove nil?))]
      (if (= 2 (count parts))
        (apply point 4326 (reverse parts))))
    s))

(defn fulltext
  "Add a where condition to a select query."
  [query & columns]
  (when-not (blank? query)
    (where `(or (~(keyword "@@")
                 (to_tsvector (array_to_string [~@columns], " "))
                 (to_tsquery ~query))
                ~@(map #(list 'like %1 (str "%" query "%")) columns))
           :and)))

(defn order-by-distance [geom-1 geom-2]
  (let [geom-1 (parse-location geom-1)
        geom-2 (parse-location geom-2)]
    (when (and geom-1 geom-2)
      (order-by `(<-> (cast ~(geometry geom-1) :geometry)
                      (cast ~(geometry geom-2) :geometry))))))

(defn order-by-bounding-box
  "Restrict the result set to rows that have `column` within
  `bounding-box`."
  [column bounding-box & [opts]]
  (when bounding-box
    (order-by
     `(st_distance
       ~column
       (st_centroid
        (st_transform
         (st_makeenvelope
          ~(.getX (.getLLB bounding-box))
          ~(.getY (.getLLB bounding-box))
          ~(.getX (.getURT bounding-box))
          ~(.getY (.getURT bounding-box))
          ~(.getSrid (.getURT bounding-box)))
         (st_srid ~column)))))))

(defn within-bounding-box
  "Restrict the result set to rows that have `column` within
  `bounding-box`."
  [column bounding-box & [opts]]
  (when bounding-box
    (let [srid (or (:srid opts) 4326)]
      (compose
       (where `(:&&
                ~column
                (cast (st_transform
                       (st_makeenvelope
                        ~(.getX (.getLLB bounding-box))
                        ~(.getY (.getLLB bounding-box))
                        ~(.getX (.getURT bounding-box))
                        ~(.getY (.getURT bounding-box))
                        ~(.getSrid (.getURT bounding-box)))
                       (st_srid ~column))
                      :geography))
              :and)
       (order-by-bounding-box column bounding-box)))))

(defn within-distance-to
  "Return a WHERE clause where `column` is within `distance`
  kilometers to `location`."
  [column location distance & [{:keys [spheroid?]}]]
  (when (and location distance)
    (sql/where
     `(st_dwithin
       (cast ~(geometry location) :geography)
       (cast ~column :geography)
       (* (cast ~(or distance 1000) :double-precision) 1000)
       ~(true? spheroid?))
     :and)))

(defn assoc-embedded [m & kv]
  (reduce
   (fn [m [k v]]
     (if-not v
       m (assoc-in m [:_embedded k] v)))
   m (partition 2 kv)))

(extend-type clojure.lang.Keyword
  IGeometry
  (geometry [k]
    k))

(defn zip-by-key
  "Zip the elements of `coll` by their `key`."
  [coll & ks]
  (zipmap (map #(get-in % ks) coll) coll))

(defn zip-by-id
  "Zip the elements of `coll` by their :id key."
  [coll]
  (zip-by-key coll :id))

(defn enabled-gdal-drivers
  "Return the short and long name of the enabled GDAL drivers."
  [db]
  (->> @(select db [(as :short_name :short-name)
                    (as :long_name :long-name)]
          (from (as '(st_gdaldrivers) :drivers))
          (order-by 1))
       (identity)))

(defn geojson-by-column
  "Return the GeoJSON of a row by a column."
  [db table geo-column id-column id-value]
  (assoc (->> @(select db [(as `(st_asgeojson ~geo-column) :json)]
                 (from table)
                 (where `(= ~id-column ~id-value)))
              first :json json/read-json)
         id-column id-value))

(defn fetch-records-by-ids [db f opts records]
  (f db (map :id records) (dissoc opts :page :per-page)))

(defn select-count-rows
  "Return a select statement that counts all rows in `table`."
  [db table]
  (sql/select db ['(count :*)]
    (sql/from table)))

(defn random-row
  "Return a random row from `table` in `db`."
  [db table]
  (first @(sql/select db [:*]
            (sql/from table)
            (sql/limit 1)
            (sql/offset `(floor (* (random) ~(select-count-rows db table)))))))

(defn dissoc-pagination
  "Remove the pagination keys :page and :per-page from `m`."
  [m]
  (dissoc m :page :per-page))
