(ns burningswell.db.weather.models
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.schemas :refer :all]
            [burningswell.db.util :refer :all]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [schema.core :as s])
  (:import sqlingvo.db.Database))

(defn- select-all [db & [opts]]
  (select db [:weather.models.id
              :weather.models.name
              :weather.models.description
              :weather.models.pattern
              :weather.models.latest-reference-time
              :weather.models.dods
              :weather.models.res-x
              :weather.models.res-y
              :weather.models.created-at
              :weather.models.updated-at]
    (from :weather.models)
    (paginate (:page opts) (:per-page opts))
    (order-by :weather.models.name)))

(s/defn all :- [WeatherModel]
  "Return all weather models in `db`."
  [db :- Database]
  @(select-all db))

(s/defn by-id :- (s/maybe WeatherModel)
  "Return the weather model in `db` by `id`."
  [db :- Database id :- s/Int]
  (first @(compose
           (select-all db)
           (where `(= :weather.models.id (cast ~id :integer))))))

(s/defn by-name :- (s/maybe WeatherModel)
  "Return the weather model in `db` by `name`."
  [db :- Database name :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :weather.models.name ~name)))))

(s/defn by-names :- [WeatherModel]
  "Return the weather models in `db` with a name in `names`."
  [db :- Database names :- [s/Str]]
  @(compose
    (select-all db)
    (where `(in :weather.models.name ~(seq names)))))

(s/defn by-names-or-all :- [WeatherModel]
  "Return the weather models in `db` with a name in `names`, or all
  weather models if `names` is empty."
  [db :- Database names :- [s/Str]]
  (if (empty? names)
    (all db) (by-names db names)))

(s/defn by-variable :- [WeatherModel]
  "Return the weather models in `db` for `variable`."
  [db :- Database variable :- WeatherVariable]
  @(compose
    (select-all db)
    (join :weather.models-variables.model-id :weather.models.id)
    (where `(= :weather.models-variables.variable-id
               ~(:id variable)))))

(s/defn akw :- (s/maybe WeatherModel)
  "Return the Regional Alaska Waters weather model."
  [db :- Database]
  (by-name db "akw"))

(s/defn nww3 :- (s/maybe WeatherModel)
  "Return the Global Wave Watch III weather model."
  [db :- Database]
  (by-name db "nww3"))
