(ns burningswell.db.natural-earth
  (:gen-class)
  (:refer-clojure :exclude [distinct replace group-by update])
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :refer [with-db]]
            [burningswell.db.spots :as spots]
            [clojure.java.io :as io]
            [clojure.string :refer [replace]]
            [clojure.tools.logging :as log]
            [commandline.core :refer [print-help with-commandline]]
            [datumbazo.core :refer :all :exclude [run with-db]]
            [datumbazo.shell :as shell]
            [datumbazo.util :as util]
            [sqlingvo.util :refer [sql-name]]
            [environ.core :refer [env]])
  (:import [org.apache.commons.io FileUtils]))

(def default-working-dir
  "The working directory for importing natural earth data."
  "natural-earth")

(def natural-earth-10m-cultural
  (str "http://www.naturalearthdata.com/http//www.naturalearthdata.com"
       "/download/10m/cultural"))

(def ^:dynamic *airports*
  {:url (str natural-earth-10m-cultural "/ne_10m_airports.zip")
   :table :natural-earth.airports
   :encoding "LATIN1"})

(def ^:dynamic *countries*
  {:url (str natural-earth-10m-cultural "/ne_10m_admin_0_countries.zip")
   :table :natural-earth.countries
   :encoding "UTF-8"})

(def ^:dynamic *ports*
  {:url (str natural-earth-10m-cultural "/ne_10m_ports.zip")
   :table :natural-earth.ports
   :encoding "UTF-8"})

(def ^:dynamic *regions*
  {:url (str natural-earth-10m-cultural "/ne_10m_admin_1_states_provinces.zip")
   :table :natural-earth.regions
   :encoding "UTF-8"})

(def ^:dynamic *time-zones*
  {:url (str natural-earth-10m-cultural "/ne_10m_time_zones.zip")
   :table :natural-earth.time-zones
   :encoding "LATIN1"})

(defmulti insert-dataset
  "Insert the Natural Earth `dataset` into the database."
  (fn [db dataset] (:table dataset)))

(defmulti update-dataset
  "Update the Natural Earth `dataset` into the database."
  (fn [db dataset] (:table dataset)))

(defn download-dataset
  "Download the Natural Earth `dataset`."
  [dataset & [opts]]
  (let [working-dir (or (:working-dir opts) default-working-dir)
        zipfile (io/file working-dir (shell/basename (:url dataset)))]
    (io/make-parents zipfile)
    (log/infof "Downloading Natural Earth dataset." )
    (log/infof "  Dataset Url....... %s" (:url dataset))
    (log/infof "  File Name......... %s" (str zipfile))
    (shell/exec-checked-script
     (format "Downloading %s to %s" (:url dataset) zipfile)
     ("wget" "-c" "-O" ~(str zipfile) ~(:url dataset)))
    (assoc dataset :zip (.getAbsolutePath zipfile))))

(defn unzip-dataset
  "Unzip the downloaded Natural Earth `dataset`."
  [dataset]
  (log/infof "Unzipping Natural Earth dataset.")
  (log/infof "  File Name......... %s" (:zip dataset))
  (shell/exec-checked-script
   (format "Unzipping %s" (:zip dataset))
   ("pushd" ~(shell/dirname (:zip dataset)))
   ("unzip" "-o" ~(shell/basename (:zip dataset)))
   ("pushd"))
  dataset)

(defn load-dataset
  "Load the Natural Earth `dataset` into the database."
  [db dataset & {:keys [entities]}]
  (let [shape (replace (:zip dataset) #"\.zip" ".dbf")
        sql (replace (:zip dataset) #"\.zip" ".sql")
        table (sql-name db (:table dataset))]
    (log/infof "Converting Natural Earth dataset shapefile to SQL.")
    (log/infof "  Shape File Name... %s" shape)
    (log/infof "  SQL File Name..... %s" sql)
    @(drop-table db [(:table dataset)]
       (if-exists true))
    (shell/shp2pgsql
     db table shape sql
     {:mode :create
      :index true
      :srid 4326
      :encoding (:encoding dataset)})
    (log/infof "Loading Natural Earth SQL file." sql table)
    (log/infof "  Database Table.... %s" (name table))
    (log/infof "  SQL File Name..... %s" sql)
    (util/exec-sql-file db sql)
    (assoc dataset :shape shape :sql sql)))

(defn import-dataset
  "Import the Natural Earth `dataset` into the database."
  [db dataset & [opts]]
  (->> (download-dataset dataset opts)
       (unzip-dataset)
       (load-dataset db)
       (update-dataset db)
       (insert-dataset db)))

(defn import-natural-earth
  "Import the Natural Earth dataset into the database."
  [db & [opts]]
  (doseq [dataset [*countries*
                   *regions*
                   *airports*
                   ;; Ports cause a segmentation fault
                   ;; *ports*
                   *time-zones*]]
    (import-dataset db dataset opts)))

;; BULK INSERT

(defmethod insert-dataset :natural-earth.airports [db dataset]
  @(insert db :airports [:country-id :region-id :name :gps-code
                         :iata-code :wikipedia-url :location]
     (select db (distinct [:r.country-id :r.id :a.name :a.gps_code
                           :a.iata_code :a.wikipedia :a.geom]
                          :on [:a.iata_code])
       (from (as :natural-earth.airports :a))
       (join (as :regions :r)
             '(on (st_intersects
                   :r.geom
                   (cast :a.geom :geography))))
       (join :airports '(on (= (lower :airports.iata-code)
                               (lower :a.iata_code)))
             :type :left)
       (where '(and (is-not-null :a.gps_code)
                    (is-not-null :a.iata_code)
                    (is-null :airports.iata-code)))))
  dataset)

(defmethod insert-dataset :natural-earth.ports [db dataset]
  @(insert db :ports [:country-id :region-id :name :type
                      :website-url :location]
     (select db (distinct [:r.country-id :r.id :p.name :p.featurecla
                           :p.website :p.geom]
                          :on [:p.name])
       (from (as :natural-earth.ports :p))
       (join (as :regions :r)
             '(on (st_intersects :r.geom (cast :p.geom :geography))))
       (join :ports
             '(on (= (lower :ports.name) (lower :p.name)))
             :type :left)
       (where '(is-null :ports.name))))
  dataset)

(defmethod insert-dataset :natural-earth.regions [db dataset]
  @(insert db :regions [:country-id :name :geom :location]
     (select db [:countries.id :n.name :n.geom '(st_centroid :n.geom)]
       (from (as :natural-earth.regions :n))
       (join :countries
             '(on (= (lower :countries.iso-3166-1-alpha-2)
                     (lower (substring :iso_3166_2 from 1 for 2)))))
       (join :regions
             '(on (and (= (lower :regions.name) (lower :n.name))
                       (= :regions.country-id :countries.id)))
             :type :left)
       (where '(and (is-not-null :n.name)
                    (is-null :regions.id)))))
  dataset)

(defmethod insert-dataset :natural-earth.time-zones [db dataset]
  @(insert db :time-zones [:geom :natural-earth-id :offset :places]
     (select db [:n.geom :n.gid :n.zone
                 '(regexp_split_to_array :n.places "\\s*,\\s*")]
       (from (as :natural-earth.time-zones :n))
       (join :time-zones
             '(on (= :time-zones.natural-earth-id :n.gid))
             :type :left)
       (where '(and (is-null :time-zones.id)
                    (is-not-null :n.gid)))))
  dataset)

(defmethod insert-dataset :default [db dataset]
  (log/warnf "No insert-dataset fn defined for %s." (:table dataset))
  dataset)

;; BULK UPDATE

(defmethod update-dataset :natural-earth.airports [db dataset]
  @(update db :airports {:country-id :u.country-id
                         :region-id :u.region-id
                         :gps-code :u.gps-code
                         :wikipedia-url :u.wikipedia
                         :location :u.geom}
           (from (as (select db (distinct
                                 [(as :r.country-id :country-id)
                                  (as :r.id :region-id)
                                  (as :a.gps_code :gps-code)
                                  (as :a.iata_code :iata-code)
                                  :a.name
                                  :a.wikipedia :a.geom]
                                 :on [:a.iata_code])
                       (from (as :natural-earth.airports :a))
                       (join (as :regions :r)
                             '(on (st_intersects
                                   :r.geom
                                   (cast :a.geom :geography))))
                       (join :airports
                             '(on (= (lower :airports.iata-code)
                                     (lower :a.iata_code)))))
                     :u))
           (where '(= :airports.iata-code :u.iata-code)))
  dataset)

(defmethod update-dataset :natural-earth.countries [db dataset]
  @(update db :countries {:geom :u.geom :location '(st_centroid :u.geom)}
           (from (as (select db [:iso_a2 :iso_a3 :iso_n3 :name :geom]
                       (from :natural-earth.countries)) :u))
           (where '(or (= (lower :countries.iso-3166-1-alpha-2)
                          (lower :u.iso_a2))
                       (= (lower :countries.iso-3166-1-alpha-3)
                          (lower :u.iso_a3))
                       (= (lower :countries.name)
                          (lower :u.name)))))
  dataset)

(defmethod update-dataset :natural-earth.ports [db dataset]
  @(update db :ports {:country-id :u.country-id
                      :region-id :u.region-id
                      :type :u.featurecla
                      :website-url :u.website
                      :location :u.geom}
           (from (as (select db (distinct [(as :r.country-id :country-id)
                                           (as :r.id :region-id)
                                           :p.name :p.featurecla :p.website
                                           :p.geom]
                                          :on [:p.name])
                       (from (as :natural-earth.ports :p))
                       (join (as :regions :r)
                             '(on (st_intersects
                                   :r.geom
                                   (cast :p.geom :geography))))
                       (join :ports
                             '(on (= (lower :ports.name)
                                     (lower :p.name)))))
                     :u))
           (where '(= (lower :ports.name) (lower :u.name))))
  dataset)

(defmethod update-dataset :natural-earth.regions [db dataset]
  @(update db :regions {:geom :u.geom :location '(st_centroid :u.geom)}
           (from (as (select db [(as :countries.id :country-id) :n.name :n.geom]
                       (from (as :natural-earth.regions :n))
                       (join :countries
                             '(on
                               (= (lower :countries.iso-3166-1-alpha-2)
                                  (lower (substring :iso_3166_2
                                                    from 1 for 2)))))
                       (join :regions
                             '(on (and (= (lower :regions.name) (lower :n.name))
                                       (= :regions.country-id :countries.id)))))
                     :u))
           (where '(and (is-not-null :u.name)
                        (= :regions.country-id :u.country-id)
                        (= (lower :regions.name) (lower :u.name)))))
  dataset)

(defmethod update-dataset :natural-earth.time-zones [db dataset]
  @(update db :time-zones {:geom :n.geom
                           :offset :n.zone
                           :places :n.places}
           (from (as (select db [:n.geom :n.gid :n.zone
                                 (as '(regexp_split_to_array
                                       :n.places "\\s*,\\s*") :places)]
                       (from (as :natural-earth.time-zones :n))
                       (join :time-zones
                             '(on (= :time-zones.natural-earth-id :n.gid))
                             :type :left)
                       (where '(and (is-null :time-zones.id)
                                    (is-not-null :n.gid))))
                     :n))
           (where '(and (is-not-null :n.gid)
                        (= :time-zones.natural-earth-id :n.gid))))
  dataset)

(defmethod update-dataset :default [db dataset]
  (log/warnf "No update-dataset fn defined for %s." (:table dataset))
  dataset)

(defn cleanup [opts]
  (let [working-dir (:working-dir opts)]
    (FileUtils/deleteDirectory (io/file working-dir))
    (log/infof "Cleaned up working directory %s." working-dir)))

(defn run
  "Import the natural earth datasets."
  [db & [opts]]
  (import-natural-earth db opts)
  (spots/set-timezone db)
  (when (:cleanup opts) (cleanup opts)))

(defn -main [& args]
  (with-commandline [[opts [cmd & args]] args]
    [[c cleanup "Cleanup after import."]
     [h help "Print this help."]]
    (if (:help opts)
      (print-help "natural-earth [OPTIONS]")
      (with-db [db (config/db env)]
        (run db {:cleanup (:cleanup opts)
                 :working-dir default-working-dir})))))

(comment (time (-main)))
