(ns burningswell.db.fixtures
  (:gen-class)
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :refer [with-db]]
            [burningswell.db.search :as search]
            [commandline.core :refer [print-help with-commandline]]
            [datumbazo.core :refer [cascade truncate with-connection]]
            [datumbazo.fixtures :as fixtures]
            [environ.core :refer [env]]
            [geo.postgis :as postgis]))

(def ^:dynamic *tables*
  [:photos
   :time-zones
   :continents
   :countries
   :regions
   :spots
   :addresses
   :airports
   :ports
   :roles
   :users
   :followings
   :sessions
   :ratings
   :roles-users
   :comments
   :photos
   :photos-spots
   :images
   :oauth.applications
   :oauth.access-tokens
   :weather.models
   :weather.variables
   :weather.models-variables
   :weather.datasets])

(def ^:dynamic *path*
  "resources/db/burningswell-db/fixtures")

(defn- dir [opts]
  (or (:directory opts) *path*))

(defn delete-fixtures
  "Delete the fixtures from the database."
  [db & [opts]]
  (doseq [table (reverse *tables*)]
    @(truncate db [table] (cascade true))))

(defn dump-fixtures
  "Dump the fixtures in the database."
  [db & [opts]]
  (fixtures/dump-fixtures db (dir opts) *tables*))

(defn load-fixtures
  "Load the fixtures into the database."
  [db & [opts]]
  (fixtures/load-fixtures db (dir opts))
  nil)

(defn show-help []
  (print-help "fixtures [delete|dump|load|]")
  (System/exit 0))

(defn run [db cmd & [opts]]
  (case (keyword cmd)
    :delete (delete-fixtures db opts)
    :dump (dump-fixtures db opts)
    :load (do (load-fixtures db opts)
              (search/refresh-views db))
    (show-help)))

(defn -main [& args]
  (with-commandline [[opts [cmd & args]] args]
    [[h help "Print this help."]
     [d directory "Read fixtures from DIRECTORY." :string "DIRECTORY"]]
    (when (:help opts) (show-help))
    (with-db [db (config/db env)]
      (with-connection [db db]
        (run db cmd opts)))))

(comment
  (-main "dump"))
