(ns burningswell.db.weather
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.util :refer :all]
            [clojure.core :as core]
            [clj-time.core :refer [now minus plus days hours]]
            [clj-time.coerce :refer [to-date-time to-sql-time]]
            [datumbazo.core :refer :all :exclude [with-db]]
            [no.en.core :refer [map-keys map-vals split-by-comma]]
            [geo.postgis :as geo]))

(require '[burningswell.db.weather :as weather])

(deftable models
  "The weather models database table."
  (table :weather.models)
  (column :id :serial :primary-key? true)
  (column :name :citext :not-null? true :unique? true)
  (column :description :text :not-null? true)
  (column :dods :text :not-null? true)
  (column :pattern :text)
  (column :res-x :float)
  (column :res-y :float)
  (column :latest-reference-time :timestamp-with-time-zone)
  (column :created-at :timestamp-with-time-zone
          :not-null? true :default '(now))
  (column :updated-at :timestamp-with-time-zone
          :not-null? true :default '(now)))

(deftable variables
  "The weather variables database table."
  (table :weather.variables)
  (column :id :serial :primary-key? true)
  (column :name :citext :not-null? true :unique? true)
  (column :description :text :not-null? true)
  (column :unit :text :not-null? true)
  (column :created-at :timestamp-with-time-zone
          :not-null? true :default '(now))
  (column :updated-at :timestamp-with-time-zone
          :not-null? true :default '(now)))

(deftable datasets
  "The weather datasets database table."
  (table :weather.datasets)
  (column :id :serial :primary-key? true)
  (column :model-id :integer :not-null? true :unique? true
          :references :weather.models/id)
  (column :variable-id :integer :not-null? true :unique? true
          :references :weather.variables/id)
  (column :reference-time :timestamp-with-time-zone :not-null? true)
  (column :valid-time :timestamp-with-time-zone :not-null? true)
  (column :das :text :not-null? true)
  (column :dds :text :not-null? true)
  (column :dods :text :not-null? true)
  (column :filename :text :not-null? true)
  (column :filesize :integer)
  (column :download-started-at :timestamp-with-time-zone)
  (column :download-finished-at :timestamp-with-time-zone)
  (column :created-at :timestamp-with-time-zone
          :not-null? true :default '(now))
  (column :updated-at :timestamp-with-time-zone
          :not-null? true :default '(now)))

(deftable region-forecasts
  "The weather forecasts database table."
  (table :weather.region-forecasts)
  (column :id :serial :primary-key? true)
  (column :region-id :integer :not-null? true :unique? true
          :references :regions/id)
  (column :variable-id :integer :not-null? true :unique? true
          :references :weather.variables/id)
  (column :valid-time :timestamp-with-time-zone :not-null? true)
  (column :avg-value :float)
  (column :min-value :float)
  (column :max-value :float)
  (column :created-at :timestamp-with-time-zone
          :not-null? true :default '(now))
  (column :updated-at :timestamp-with-time-zone
          :not-null? true :default '(now)))

(deftable spot-forecasts
  "The weather forecasts database table."
  (table :weather.spot-forecasts)
  (column :id :serial :primary-key? true)
  (column :spot-id :integer :not-null? true :unique? true
          :references :spots/id)
  (column :model-id :integer :not-null? true :unique? true
          :references :weather.models/id)
  (column :variable-id :integer :not-null? true :unique? true
          :references :weather.variables/id)
  (column :reference-time :timestamp-with-time-zone :not-null? true)
  (column :valid-time :timestamp-with-time-zone :not-null? true)
  (column :value :float)
  (column :created-at :timestamp-with-time-zone
          :not-null? true :default '(now))
  (column :updated-at :timestamp-with-time-zone
          :not-null? true :default '(now)))

(defquery1 dataset
  "Find the dataset by model, variable and valid time."
  [db dataset]
  (select db [:weather.datasets.*]
    (from :weather.datasets)
    (where `(and (= :weather.datasets.model-id
                    ~(:model-id dataset))
                 (= :weather.datasets.variable-id
                    ~(:variable-id dataset))
                 (= :weather.datasets.valid-time
                    ~(to-date-time (:valid-time dataset)))))))

(defn dataset-by-req
  "Returns the dataset for the Pedestal request `req`."
  [req]
  (dataset-by-id (:db req) (-> req :path-params :id)))

(defn model-by-req
  "Returns the model for the Pedestal request `req`."
  [req]
  (model-by-id (:db req) (-> req :path-params :id)))

(defn variable-by-req
  "Returns the variable for the Pedestal request `req`."
  [req]
  (variable-by-id (:db req) (-> req :path-params :id)))

(defquery models
  "Returns all weather models"
  [db & [opts]]
  (let [{:keys [query page per-page]} opts]
    (select db (remove :hidden? (columns models-table))
      (from :weather.models)
      (fulltext query :weather.models.name :weather.models.description)
      (order-by :weather.models.name)
      (paginate page per-page))))

(defquery variables
  "Returns all weather variables"
  [db & [opts]]
  (let [{:keys [query page per-page]} opts]
    (select db (remove :hidden? (columns variables-table))
      (from :weather.variables)
      (fulltext query :weather.variables.name :weather.variables.description)
      (order-by :weather.variables.name)
      (paginate page per-page))))

(defquery datasets-by-model
  "Returns the datasets by `model`."
  [db model & [opts]]
  (select db [:weather.datasets.*]
    (from :weather.datasets)
    (where `(= :weather.datasets.model-id ~(:id model)))))

(defquery datasets-by-variable
  "Returns the datasets by `variable`."
  [db variable & [opts]]
  (select db [:weather.datasets.*]
    (from :weather.datasets)
    (where `(= :weather.datasets.variable-id ~(:id variable)))))

(defquery delete-dataset
  "Delete the `dataset` from the database."
  [db dataset]
  (delete db :weather.datasets
    (where `(and (= :model-id ~(:model-id dataset))
                 (= :variable-id ~(:variable-id dataset))
                 (= :valid-time ~(:valid-time dataset))))))

(defquery spot-forecasts-by-spot
  "Returns the forecasts of `spot`."
  [db spot & {:keys [start end]}]
  (select db [:spot-id :valid-time :variables.name :value]
    (from :weather.spot-forecasts)
    (join :weather.variables.id :weather.spot-forecasts.variable-id)
    (where `(= :spot-id ~(:id spot)))
    (if-let [start (to-date-time start)]
      (where `(>= :valid-time ~start) :and))
    (if-let [end (to-date-time end)]
      (where `(< :valid-time ~end) :and))
    (order-by 1)))

(defquery models-by-variable
  "Returns the models by `variable`."
  [db variable & [opts]]
  (select db [:weather.models.*]
    (from :weather.models-variables :weather.models)
    (where `(and (= :weather.models-variables.model-id
                    :weather.models.id)
                 (= :weather.models-variables.variable-id
                    ~(:id variable))))))

(defquery variables-by-model
  "Returns the variables by `model`."
  [db model & [opts]]
  (select db [:weather.variables.*]
    (from :weather.models-variables :weather.variables)
    (where `(and (= :weather.models-variables.model-id
                    ~(:id model))
                 (= :weather.models-variables.variable-id
                    :weather.variables.id)))))

(defn akw
  "Returns the Regional Alaska Waters Wave Model model."
  [db] (model-by-name db "akw"))

(defn nww3
  "Returns the global Wave Watch III model."
  [db] (model-by-name db "nww3"))

(defn gfs
  "Returns the Global Forecast System model."
  [db] (model-by-name db "gfs"))

(defn htsgwsfc
  "Returns the significant height of combined wind waves and swell variable."
  [db] (variable-by-name db "htsgwsfc"))

(defn dirpwsfc
  "Returns the primary wave direction variable."
  [db] (variable-by-name db "dirpwsfc"))

(defn tcdcclm
  "Returns the total cloud cover variable."
  [db] (variable-by-name db "tcdcclm"))

(defn make-dataset
  "Make a weather dataset."
  [model variable reference-time]
  (-> {:model-id (:id model)
       :variable-id (:id variable)
       :reference-time reference-time}
      (assoc-embedded
       :model model :variable variable)))

(defquery models-in-names
  "Returns weather models by their `names`."
  [db names]
  (select db [:weather.models.*]
    (from :weather.models)
    (where `(in :name ~(seq names)))
    (order-by :name)))

(defquery variables-in-names
  "Returns weather variables by their `names`."
  [db names]
  (let [names (split-by-comma names)]
    (select db [:weather.variables.*]
      (from :weather.variables)
      (where `(in :name ~(seq names)))
      (order-by :name))))

(defn model-with-variables
  "Returns `model` with it's variables."
  [db model] (assoc-embedded model :variables (variables-by-model db model)))

(defn models-with-variables
  "Returns `models` with it's variables."
  [db & [models]]
  (map (partial model-with-variables db)
       (or models (burningswell.db.weather/models db))))

(defquery1 dataset
  "Find the weather dataset by `model`, `variable`, `reference-time`
  and `valid-time`."
  [db dataset]
  (select db [:weather.datasets.*]
    (from :weather.datasets)
    (where `(and (= :weather.datasets.model-id ~(:model-id dataset))
                 (= :weather.datasets.variable-id ~(:variable-id dataset))
                 (= :weather.datasets.reference-time ~(:reference-time dataset))
                 (= :weather.datasets.valid-time ~(:valid-time dataset))))))

;; IMAGE

(defn make-viewer []
  (let [frame (javax.swing.JFrame.)]
    (fn [image]
      (let [pane (.getContentPane frame)
            label (javax.swing.JLabel. image)]
        (.removeAll pane)
        (.add pane label)
        (.pack frame)
        (if-not (.isVisible frame)
          (.setVisible frame true))))))

(defn write-bytes [filename bytes]
  (with-open [f (java.io.FileOutputStream. filename)]
    (.write f bytes)))

(defn write-png [db filename query]
  (write-bytes filename (:st_aspng (first (run db query)))))

(defn query-image [db query]
  (javax.swing.ImageIcon. (:st_aspng (first (run db query)))))

(defn max-dataset-valid-time
  "Select the maximum valid time of the datasets."
  [db & [time]]
  (->> @(select db ['(max :valid-time)]
          (from :weather.datasets)
          (when time (where `(<= :valid-time ~time))))
       first :max to-date-time))

(defn min-dataset-valid-time
  "Select the minimum valid time of the datasets."
  [db & [time]]
  (->> @(select db ['(min :valid-time)]
          (from :weather.datasets)
          (when time (where `(>= :valid-time ~time))))
       first :min to-date-time))

(defn select-forecast-timestamps
  "Return forecasts timestamps between `start` and `end`."
  [db & [{:keys [start end]}]]
  (let [start (or (min-dataset-valid-time db start) (now))
        end (or (max-dataset-valid-time db end) (plus start (days 14)))]
    (select db [:datasets.model-id
                :datasets.variable-id
                (as '(max :datasets.reference-time) :reference-time)
                (as '(min :datasets.valid-time) :valid-time)
                :timestamps.time]
      (from (as (select db [(as :n :time)]
                  (from (as `(generate_series
                              ~start ~end
                              (cast "1 hours" :interval)) :n)))
                :timestamps))
      (join :weather.datasets
            '(on (overlaps
                  (range (- :timestamps.time
                            (cast "1.5 hours" :interval))
                         (+ :timestamps.time
                            (cast "1.5 hours" :interval)))
                  (range (- :datasets.valid-time
                            (cast "1.5 hours" :interval))
                         (+ :datasets.valid-time
                            (cast "1.5 hours" :interval))))))
      (join :weather.rasters.dataset-id :weather.datasets.id)
      (where `(and (>= :timestamps.time ~start)
                   (<= :timestamps.time ~end)))
      (group-by :datasets.model-id :datasets.variable-id :timestamps.time)
      (order-by :datasets.model-id :datasets.variable-id :timestamps.time))))

(defn spot-forecasts
  "Returns the spot forecasts from the raw raster data."
  [db & [{:keys [spots start end] :as opts}]]
  (select db (distinct
              [(as :spots.id :spot-id)
               :datasets.model-id
               :datasets.variable-id
               :datasets.reference-time
               (as :timestamps.time :valid-time)
               (as `(avg_neighborhood
                     (st_neighborhood
                      :rast 1 (cast :location :geometry) 1 1))
                   :value)]
              :on [:spots.id
                   :datasets.variable-id
                   :timestamps.time])
    (from (as (select-forecast-timestamps db opts) :timestamps))
    (join :weather.datasets
          '(on (and (= :timestamps.model-id :datasets.model-id)
                    (= :timestamps.variable-id :datasets.variable-id)
                    (= :timestamps.reference-time :datasets.reference-time)
                    (= :timestamps.valid-time :datasets.valid-time))))
    (join :weather.rasters.dataset-id :weather.datasets.id)
    (join :spots `(on (st_intersects :rast (cast :location :geometry))))
    (when (not-empty spots)
      (where `(in :spots.id ~(map :id spots)) :and))
    (order-by :spots.id :datasets.variable-id :timestamps.time
              (desc '(abs (* (st_scalex :rast)
                             (st_scaley :rast)))))))

(defn import-spot-forecasts
  "Save the weather forecasts by first updating existing forecasts and
  then inserting new forecast, optionaly between `start` and `end`."
  [db & [{:keys [end radius spots start] :as opts}]]
  (insert db :weather.spot-forecasts
      [:spot-id :model-id :variable-id :reference-time :valid-time :value]
    (spot-forecasts db opts)
    (on-conflict [:spot-id :variable-id :valid-time]
      (do-update
       {:model-id :EXCLUDED.model-id
        :reference-time :EXCLUDED.reference-time
        :value :EXCLUDED.value}))))

(defn- reduce-weather [forecasts]
  (reduce
   #(assoc %1
           (keyword (:name %2)) (:value %2)
           :spot-id (:spot-id %2)
           :valid-time (:valid-time %2))
   {} forecasts))

(defquery current-spot-weather
  "Returns the current weather at `spot`."
  [db spot]
  (select db [:*]
    (from :weather.current-forecasts)
    (where `(= :spot-id ~(:id spot)))))

(defquery current-spots-weather
  "Returns the current weather at `spots`."
  [db spots]
  (let [spot-ids (apply list (map :id spots))]
    (select db [:current-forecasts.*
                (as :models.name :model)
                (as :variables.name :variable)]
      (from :weather.current-forecasts)
      (join :weather.variables.id :current-forecasts.variable-id)
      (join :weather.models.id :current-forecasts.model-id)
      (where `(:in :spot-id ~(if (empty? spot-ids)
                               (list nil) spot-ids))))))

(defn weather-variable-names
  "Returns all weather variable names."
  [db]
  (->> (select db [:name] (from :weather.variables))
       (run)
       (map :name)))

(defquery spot-weather
  "Returns the weather at `spots`."
  [db spots & [{:keys [variables start end units page per-page] :as opts}]]
  (let [units (or units "eu")
        spots (if (sequential? spots) spots [spots])
        spot-ids (apply list (map :id spots))
        variable-names (cond
                         (string? variables)
                         (split-by-comma variables)
                         :else (weather-variable-names db))
        start (or (to-date-time start) (minus (now) (hours 3)))
        end (or (to-date-time end) (plus start (days 14)))]
    (select db [(as `(convert-value :forecasts.value :forecasts.unit ~units)
                    :value)
                (as `(convert-unit :forecasts.unit ~units) :unit)
                :forecasts.valid-time
                (as :models.name :model)
                (as :variables.name :variable)]
      (from (as (case (keyword (:group-by opts))
                  :current :weather.current-forecasts
                  :hourly :weather.spot-forecasts
                  :hourly-3 :weather.spot-weather-3-hours
                  :daily :weather.daily-spot-forecasts
                  :weather.spot-weather-3-hours)
                :forecasts))
      (join :weather.variables.id :forecasts.variable-id)
      (join :weather.models.id :forecasts.model-id)
      (where `(and (:in :spot-id ~spot-ids)
                   (:in :variables.name ~(seq variable-names))
                   (>= :valid-time ~start)
                   (<= :valid-time ~end)))
      (order-by :spot-id :valid-time))))

(defn weather-report [weather]
  (->> (core/group-by :valid-time weather)
       (map (fn [[valid-time rows]]
              [valid-time
               (reduce
                (fn [result row]
                  (assoc result
                         (keyword (:variable row))
                         (select-keys row [:value :unit])))
                {} rows)]))
       (into {})))

(defn spot-weather-reports
  [db spots & [opts]]
  (let [weather (spot-weather db spots opts)
        weather (core/group-by :spot-id weather)
        htsgwsfc (spot-weather
                  db spots
                  {:group-by :hourly-3 :variables [{:name "htsgwsfc"}]})
        htsgwsfc (core/group-by :spot-id htsgwsfc)]
    (map (fn [spot]
           (-> (assoc-in spot [:_embedded :weather]
                         (weather-report (get weather (:id spot))))
               (assoc-in [:_embedded :htsgwsfc]
                         (map #(select-keys
                                %1 [:unit :value :valid-time :variable])
                              (get htsgwsfc (:id spot))))))
         (if (sequential? spots) spots [spots]))))

(defn weather-by-spot
  [db spot & [opts]]
  (weather-report (spot-weather db [spot] opts)))

(defn wave-heights-by-spot
  [db spot & [opts]]
  (let [start (or (:start opts) (minus (now) (hours 3)))
        end (:end opts)
        units (or (:units opts) "eu")]
    @(select db [:forecasts.valid-time
                 (as `(convert-value :forecasts.value :forecasts.unit ~units)
                     :value)
                 (as `(convert-unit :forecasts.unit ~units) :unit)
                 (as :models.name :model)
                 (as :variables.name :variable)]
       (from (as :weather.spot-weather-3-hours :forecasts))
       (join :weather.variables.id :forecasts.variable-id)
       (join :weather.models.id :forecasts.model-id)
       (where `(and (= :spot-id ~(:id spot))
                    (= :variables.name "htsgwsfc")
                    (>= :valid-time ~start)
                    ~(if end `(<= :valid-time ~end) true)))
       (order-by :valid-time))))

(defn current-weather-by-spots
  "Returns `spots` with their weather forecast."
  [db spots & [opts]]
  (let [start (minus (now) (hours 3))
        end (plus start (hours 3))
        units (or (:units opts) "eu")]
    (->> @(select db [(distinct
                       [:spot-id
                        :valid-time
                        (as `(convert-value :value :unit ~units) :value)
                        (as `(convert-unit :unit ~units) :unit)
                        (as :models.name :model)
                        (as :variables.name :variable)]
                       :on [:spot-id :variables.name])]
            (from :weather.spot-forecasts)
            (join :weather.variables.id :spot-forecasts.variable-id)
            (join :weather.models.id :spot-forecasts.model-id)
            (where `(and (in :spot-id ~(map :id spots))
                         (>= :valid-time ~start)
                         (<= :valid-time ~end)))
            (order-by :spot-id :variables.name))
         (core/group-by :spot-id)
         (map-vals weather-report))))

(defn current-weather-by-spot
  "Returns the `spot` with it's weather forecast."
  [db spot]
  (first (spot-weather-reports db [spot] {:group-by :current})))

(defn refresh-3-hourly-spot-weather
  "Refresh the :weather.spot-weather-3-hours materialized view."
  [db]
  (refresh-materialized-view db :weather.spot-weather-3-hours))

(defquery svm-features
  "Returns the SVM weather features."
  [db & [{:keys [spots start end page per-page] :as opts}]]
  (select db [:*]
    (from :weather.svm-features)
    (if-let [start (to-date-time start)]
      (where `(>= :valid-time ~start) :and))
    (if-let [end (to-date-time end)]
      (where `(<= :valid-time ~end) :and))
    (if-not (empty? spots)
      (where `(:in :spot-id ~(map :id spots)) :and))))

(defn create-model
  "Create a new weather model."
  [db model]
  (let [row (insert-model db model)]
    (model-by-id db (:id row))))

(defn create-variable
  "Create a new weather variable."
  [db variable]
  (let [row (insert-variable db variable)]
    (variable-by-id db (:id row))))

(defn create-dataset
  "Create a new weather dataset."
  [db dataset]
  (let [row (insert-dataset db dataset)]
    (dataset-by-id db (:id row))))

(defn refresh-views
  "Refresh the materialized views."
  [db]
  @(refresh-3-hourly-spot-weather db))

(defn- select-current-forecast-time
  "Return the current forecast time."
  [db interval]
  (select db [(as '(min :datasets.valid-time) :valid-time)]
    (from :weather.datasets)
    (where `(>= :datasets.valid-time
                (- (now)
                   (cast ~(or interval "3 hours") :interval))))
    (group-by :datasets.variable-id)))

(defn select-weather-forecast [db location & [opts]]
  (let [start (to-date-time (or (:start opts) (now)))
        end (to-date-time (or (:end opts) (plus start (hours 3))))]
    (select db (distinct
                [(as '(json_build_object
                       "id" :models.id
                       "name" :models.name
                       "description" :models.description
                       "reference-time" :datasets.reference-time)
                     :model)
                 (as `(json_build_object
                       "id" :variables.id
                       "name" :variables.name
                       "description" :variables.description
                       "valid-time" :datasets.valid-time
                       "value" (round (cast (avg_neighborhood
                                             (st_neighborhood
                                              :rast 1 ~location 1 1))
                                            :numeric)
                                      4)
                       "unit" :variables.unit)
                     :variable)]
                :on [:datasets.variable-id :datasets.valid-time])
      (from :weather.datasets)
      (join :weather.rasters.dataset-id :weather.datasets.id)
      (join :weather.variables.id :weather.datasets.variable-id)
      (join :weather.models.id :weather.datasets.model-id)
      (where `(and (st_intersects :rast ~location)
                   (>= :datasets.valid-time ~start)
                   (< :datasets.valid-time ~end)))
      (when-let [variables (not-empty (:variables opts))]
        (where `(in :weather.variables.name ~(seq variables)) :and))
      (order-by :datasets.variable-id :datasets.valid-time
                (desc '(abs (* (st_scalex :rast)
                               (st_scaley :rast))))))))

(defn weather-forecast [db location & [opts]]
  (->> @(select-weather-forecast db location opts)
       (map #(update-in % [:model :reference-time] to-date-time))
       (map #(update-in % [:variable :valid-time] to-date-time))
       (clojure.core/group-by #(get-in % [:variable :valid-time]))))

(defn select-weather-forecast [db location & [opts]]
  (let [start (to-date-time (or (:start opts) (.toDateMidnight (now))))
        end (to-date-time (or (:end opts) (plus start (hours 24))))]
    (select db (distinct
                [(as :models.name :model)
                 (as :variables.name :variable)
                 :variables.unit
                 :datasets.reference-time
                 :datasets.valid-time
                 (as `(cast (avg_neighborhood
                             (st_neighborhood
                              :rast 1 ~location 1 1))
                            :double-precision) :value)
                 ;; (as '(json_build_object
                 ;;       "id" :models.id
                 ;;       "name" :models.name
                 ;;       "description" :models.description
                 ;;       "reference-time" :datasets.reference-time)
                 ;;     :model)
                 ;; (as `(json_build_object
                 ;;       "id" :variables.id
                 ;;       "name" :variables.name
                 ;;       "description" :variables.description
                 ;;       "valid-time" :datasets.valid-time
                 ;;       "value" (round (cast (avg_neighborhood
                 ;;                             (st_neighborhood
                 ;;                              :rast 1 ~location 1 1))
                 ;;                            :numeric)
                 ;;                      4)
                 ;;       "unit" :variables.unit)
                 ;;     :variable)
                 ]
                :on [:datasets.variable-id :datasets.valid-time])
      (from :weather.datasets)
      (join :weather.rasters.dataset-id :weather.datasets.id)
      (join :weather.variables.id :weather.datasets.variable-id)
      (join :weather.models.id :weather.datasets.model-id)
      (where `(and (st_intersects :rast ~location)
                   (>= :datasets.valid-time ~start)
                   (< :datasets.valid-time ~end)))
      (when-let [variables (not-empty (:variables opts))]
        (where `(in :weather.variables.name ~(seq variables)) :and))
      (order-by :datasets.variable-id :datasets.valid-time
                (desc '(abs (* (st_scalex :rast)
                               (st_scaley :rast))))))))

(defn weather-forecast [db location & [opts]]
  (->> @(select-weather-forecast db location opts)
       ;; (map #(update-in % [:model :reference-time] to-date-time))
       ;; (map #(update-in % [:variable :valid-time] to-date-time))
       (clojure.core/group-by #(select-keys % [:variable :valid-time]))))

(defn weather-forecast [db location & [opts]]
  (reduce (fn [forecast row]
            (assoc-in forecast [(keyword (:variable row)) (:valid-time row)]
                      (dissoc row :variable :valid-time)))
          {} @(select-weather-forecast db location opts)))

(comment
  (clojure.pprint/pprint
   (weather-forecast
    db mundaka
    {:variables ["htsgwsfc" "windsfc"]
     :start #inst "2016-07-20"
     :end #inst "2016-07-22"
     })))

(comment
  (clojure.pprint/print-table
   @(select-weather-forecast
     db mundaka
     {:variables ["htsgwsfc" "windsfc"]
      :start #inst "2016-07-20"
      :end #inst "2016-07-22"})))

(comment

  (def mundaka (geo/point 4326 -2.69500976358495 43.40921730553149))
  (def pipeline (geo/point 4326 -158.0512047 21.6650562))

  (def db reloaded.repl/system)

  (time (weather-forecast db mundaka))

  (time (weather-forecast
         db pipeline {:start #inst "2016-07-20"
                      :end #inst "2016-07-22"}))

  (clojure.pprint/pprint (weather-forecast db mundaka))

  (require '[burningswell.config.core :as config])
  (require '[burningswell.db.connection :refer [with-db]])
  (require '[environ.core :refer [env]])


  (with-db [db (config/db env)]
    (wave-heights-by-spot db {:id 1}))

  (def db (component/start
           (burningswell.db.connection/new-db
            (burningswell.config.core/db environ.core/env))))



  (time
   @(import-spot-forecasts
     (-> reloaded.repl/system :db)
     {:spots [{:id 2556} {:id 2128} {:id 4569}]}))

  (clojure.pprint/print-table
   (->> @(spot-forecasts
          reloaded.repl/system
          {:spots [{:id 4569}]})
        ;; (take 20)
        ))

  (refresh-views (-> reloaded.repl/system :db))

  (min-dataset-valid-time db)
  (max-dataset-valid-time db)

  (clojure.pprint/print-table
   @(spot-forecasts db {:spots [{:id 2}]}))



  (first (spot-forecasts db :spots [{:id 2}]))
  @(import-spot-forecasts db)
  (svm-features* (db))
  (map :spot-id (svm-features (db)))
  (svm-features* (db) {:spots [{:id 1}]})
  (time (prn (first (svm-features (db)))))
  (time (prn (first (svm-features (db) {:spots [{:id 1}]}))))
  (map :spot-id (svm-features (db) {:spots [{:id 12}]}))
  (time (import-spot-forecasts (db))))
