(ns burningswell.db.addresses
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.countries :as countries]
            [burningswell.db.schemas :refer :all]
            [burningswell.db.regions :as regions]
            [burningswell.db.util :refer :all]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [geo.core :refer [srid point-x point-y]]
            [geo.postgis :refer [geometry point]]
            [schema.core :as s])
  (:import sqlingvo.db.Database
           org.postgis.Point))

(defn- select-all
  "Returns all addresses."
  [db & [opts]]
  (let [{:keys [query page per-page]} opts]
    (select db [:addresses.id
                :addresses.formatted
                :addresses.street-name
                :addresses.street-number
                :addresses.postal-code
                :addresses.city
                (as '(cast :addresses.location :geometry) :location)
                :addresses.created-at
                :addresses.updated-at
                (as `(json_build_object
                      "country" (json-embed-country :countries)
                      "region" (json-embed-region :regions)
                      "user" (json-embed-user :users))
                    :_embedded)]
      (from :addresses)
      (join :countries.id :addresses.country-id)
      (join :regions.id :addresses.region-id :type :left)
      (join :users.id :addresses.user-id :type :left)
      (fulltext query :addresses.formatted)
      (cond
        (:location opts)
        (order-by-distance :addresses.location (:location opts))
        :else (order-by :addresses.formatted))
      (paginate page per-page))))

(defn- row [spot]
  (assoc (select-keys spot [:formatted :city :location :street-name
                            :street-number :postal-code])
         :country-id (-> spot :_embedded :country :id)
         :region-id (-> spot :_embedded :region :id)
         :user-id (-> spot :_embedded :user :id)))

(s/defn all :- [Address]
  "Return all addresses in `db`."
  [db :- Database & [opts]]
  @(select-all db opts))

(s/defn by-id :- (s/maybe Address)
  "Return the country in `db` by `id`."
  [db :- Database id :- s/Num]
  (first @(compose
           (select-all db)
           (where `(= :addresses.id ~id)))))

(s/defn by-formatted :- (s/maybe Address)
  "Return the country in `db` by `formatted`."
  [db :- Database formatted :- s/Str]
  (first @(compose
           (select-all db)
           (where `(= :addresses.formatted ~formatted)))))

(s/defn by-location :- (s/maybe Address)
  "Return the country in `db` by `location`."
  [db :- Database location :- Point]
  (first @(compose
           (select-all db)
           (where `(= :addresses.location ~location)))))

(s/defn delete
  "Delete `address` from `db`."
  [db :- Database address :- Address]
  (->> @(sql/delete db :addresses
          (where `(= :addresses.id
                     ~(:id address))))
       first :count))

(s/defn insert :- Address
  "Insert `address` into `db`."
  [db :- Database address]
  (->> @(sql/insert db :addresses []
          (values [(row address)])
          (returning :id))
       first :id (by-id db)))

(s/defn update :- Address
  "Update `address` in `db`."
  [db :- Database address]
  (->> @(sql/update db :addresses
          (row address)
          (where `(or (= :addresses.id ~(:id address))
                      (= :addresses.location ~(:location address))))
          (returning :id))
       first :id (by-id db)))

(s/defn ke-nui-road-haleiwa
  "Return the address 59-343 Ke Nui Road, Haleiwa, HI 96712, USA."
  [db :- Database]
  (by-id db 7))
