(ns burningswell.db.search
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.countries :as countries]
            [burningswell.db.regions :as regions]
            [burningswell.db.schemas :refer :all]
            [burningswell.db.spots :as spots]
            [burningswell.db.util :refer :all]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [schema.core :as s])
  (:import sqlingvo.db.Database))

(s/defn autocomplete :- [s/Any]
  "Return all comments in `db`."
  [db :- Database & [opts]]
  (let [{:keys [page per-page query]} opts]
    (->> @(select db [:* (as `(cast
                               (ts_rank_cd
                                (to_tsvector ~query)
                                (plainto_tsquery ~query))
                               :decimal)
                             :rank)]
            (from :search.autocomplete)
            (fulltext query :search.autocomplete.term)
            (paginate page per-page))
         (map #(update-in % [:type] keyword)))))

(defn- filter-by-type [type results]
  (let [type (keyword type)]
    (filter #(= type (:type %)) results)))

(defn- collect-by-type [db collect-fn type results]
  (->> (filter-by-type type results)
       (map :id)
       (collect-fn db)
       (zip-by-id)))

(defn- countries [db results]
  (collect-by-type db countries/by-ids :country results))

(defn- regions [db results]
  (collect-by-type db regions/by-ids :region results))

(defn- spots [db results]
  (collect-by-type db spots/by-ids :spot results))

(s/defn details :- s/Any
  "Return all comments in `db`."
  [db :- Database & [opts]]
  (let [results (autocomplete db opts)
        countries (countries db results)
        regions (regions db results)
        spots (spots db results)]
    (map (fn [{:keys [id rank type] :as result}]
           (-> (get (case type
                      :country countries
                      :region regions
                      :spot spots)
                    id)
               (assoc :type type :rank rank)))
         results)))

(s/defn refresh-views :- [s/Any]
  "Refresh the search views."
  [db :- Database]
  @(refresh-materialized-view db :search.autocomplete
     (concurrently true)))
