(ns burningswell.db.oauth
  (:refer-clojure :exclude [distinct group-by update])
  (:require [clojure.java.jdbc :as jdbc]
            [datumbazo.core :refer :all]))

(deftable oauth-applications
  "The OAuth applications database table."
  (table :oauth.applications)
  (column :id :serial :primary-key? true)
  (column :user-id :integer :references :users/id)
  (column :name :citext :not-null? true :unique? true)
  (column :description :text :not-null? true)
  (column :client-id :uuid :not-null? true :unique? true)
  (column :client-secret :text :not-null? true)
  (column :redirect-uri :text :not-null? true)
  (column :created-at :timestamp-with-time-zone
          :not-null? true :default '(now))
  (column :updated-at :timestamp-with-time-zone
          :not-null? true :default '(now)))

(deftable oauth-access-grants
  "The OAuth access grants database table."
  (table :oauth.access-grants)
  (column :id :serial :primary-key? true)
  (column :application-id :integer :references :oauth.applications/id)
  (column :user-id :integer :references :users/id)
  (column :token :text :not-null? true :unique? true)
  (column :redirect-uri :text :not-null? true)
  (column :expires-at :timestamp-with-time-zone)
  (column :revoked-at :timestamp-with-time-zone)
  (column :created-at :timestamp-with-time-zone
          :not-null? true :default '(now))
  (column :updated-at :timestamp-with-time-zone
          :not-null? true :default '(now)))

(deftable oauth-access-tokens
  "The OAuth access tokens database table."
  (table :oauth.access-tokens)
  (column :id :serial :primary-key? true)
  (column :application-id :integer :references :oauth.applications/id)
  (column :user-id :integer :references :users/id)
  (column :access-token :text :not-null? true :unique? true)
  (column :refresh-token :text :unique? true)
  (column :expires-at :timestamp-with-time-zone)
  (column :revoked-at :timestamp-with-time-zone)
  (column :created-at :timestamp-with-time-zone
          :not-null? true :default '(now))
  (column :updated-at :timestamp-with-time-zone
          :not-null? true :default '(now)))

(defquery authorized-for
  "Returns the authorized OAuth application of `user`."
  [db user]
  (select db (columns oauth-applications-table)
    (from :oauth.applications)
    (join :oauth.access-tokens.application-id
          :oauth.applications.id)
    (where `(and (= :oauth.access-tokens.user-id ~(:id user))
                 (= :oauth.access-tokens.revoked-at ~nil)))
    (apply group-by (columns oauth-applications-table))))

(defn- ensure-application-by-name
  "Returns the OAuth application by name and raises an exception if
  the application could not found."
  [db name]
  (let [app (oauth-application-by-name db name)]
    (if-not app
      (throw (ex-info "Can't find OAuth application." {:name name}))
      app)))

(defn burningswell-application
  "Returns the Burning Swell OAuth application."
  [db] (ensure-application-by-name db "Burning Swell"))

(defn facebook-application
  "Returns the Facebook OAuth application."
  [db] (ensure-application-by-name db "Facebook"))

(defn twitter-application
  "Returns the Twitter OAuth application."
  [db] (ensure-application-by-name db "Twitter"))
