(ns burningswell.db.images
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.schemas :refer :all]
            [burningswell.db.util :refer :all]
            [burningswell.flickr :as flickr]
            [clojure.core :as core]
            [clojure.java.jdbc :as jdbc]
            [clojure.set :refer [rename-keys]]
            [datumbazo.core :as sql :exclude [delete insert update] :refer :all]
            [geo.core :refer [point-x point-y point]]
            [hal.core :as hal]
            [schema.core :as s])
  (:import sqlingvo.db.Database))

(def embedded-keys
  "The keys for an embedded image."
  [:id :content-length :content-type :height :label :url
   :width :created-at :updated-at])

(defn- row [image]
  (assoc (select-keys image [:content-disposition :content-length :content-md5
                             :content-type :width :height :label :s3-key :url])
         :photo-id (-> image :_embedded :photo :id)))

(defn- select-all [db & [opts]]
  (select db [:images.id
              :images.content-disposition
              :images.content-length
              :images.content-md5
              :images.content-type
              :images.width
              :images.height
              :images.label
              :images.s3-key
              :images.url
              :images.created-at
              :images.updated-at
              (as `(json_build_object
                    "photo"
                    (json_build_object
                     "id" :images.photo-id))
                  :_embedded)]
    (from :images)
    (order-by (or (:order-by opts)
                  (desc :images.width)))
    (paginate (:page opts) (:per-page opts))))

(s/defn all :- [Image]
  "Return all images in `db`."
  [db :- Database & [opts]]
  @(select-all db opts))

(s/defn by-id :- (s/maybe Image)
  "Return the country in `db` by `id`."
  [db :- Database id :- s/Num]
  (first @(compose
           (select-all db)
           (where `(= :images.id ~id)))))

(s/defn by-photo
  "Return the images of `photo` from `db`."
  [db :- Database photo :- Photo & [opts]]
  @(compose
    (select-all db opts)
    (where `(= :images.photo-id ~(:id photo)))))

(s/defn by-photos
  "Return the images of `photos` from `db`."
  [db :- Database photos :- [Photo] & [opts]]
  @(compose
    (select-all db opts)
    (where `(in :images.photo-id ~(map :id photos)))))

(s/defn delete
  "Delete `image` from `db`."
  [db :- Database image :- Image]
  (->> @(sql/delete db :images
          (where `(= :images.id
                     ~(:id image))))
       first :count))

(s/defn insert
  "Insert `image` into `db`."
  [db :- Database image]
  (->> @(sql/insert db :images []
          (values (row image))
          (returning :id))
       first :id (by-id db)))

(s/defn update
  "Update `image` in `db`."
  [db :- Database image]
  (some->> @(sql/update db :images (row image)
              (where `(or (= :images.id ~(:id image))
                          (and (= :images.photo-id
                                  ~(-> image :_embedded :photo :id))
                               (= :images.label ~(:label image)))))
              (returning :id))
           first :id (by-id db)))

(s/defn save
  "Save `image` to `db`."
  [db :- Database image]
  (or (update db image)
      (insert db image)))

(defn select-embedded
  "Return an embedded version of `image`."
  [image]
  (select-keys image embedded-keys))

(defn photo-id
  "Return the photo-id of `image`."
  [image]
  (get-in image [:_embedded :photo :id]))

(defn group-by-photo-id
  "Group `images` by their photo."
  [images]
  (core/group-by photo-id images))

(defn zip-by-label
  "Zip the `images` by their label."
  [images]
  (zipmap (map (comp keyword :label) images) images))
