(ns burningswell.db.ratings
  (:refer-clojure :exclude [distinct group-by update])
  (:require [clj-time.core :refer [now minus millis years plus]]
            [clj-time.coerce :refer [from-long to-long to-date]]
            [clojure.java.jdbc :as jdbc]
            [datumbazo.core :refer :all]))

(deftable ratings
  "The spot ratings database table."
  (column :id :serial :primary-key? true)
  (column :user-id :integer :not-null? true :references :users/id)
  (column :spot-id :integer :not-null? true :references :spots/id)
  (column :rating :integer :not-null? true)
  (column :rated-at :timestamp-with-time-zone :not-null? true :default '(now))
  (column :created-at :timestamp-with-time-zone :not-null? true :default '(now))
  (column :updated-at :timestamp-with-time-zone
          :not-null? true :default '(now)))

(defquery ratings-by-spot
  "Returns the ratings for the given spot."
  [db spot & [opts]]
  (let [{:keys [page per-page]} opts]
    (select db [:ratings.*]
      (from :ratings)
      (where `(= :spot-id ~(:id spot)))
      (paginate page per-page)
      (order-by :created-at))))

(defn rand-time
  "Returns a random time between `start` and `end`."
  [start end]
  (let [start (to-long start)
        end (to-long end)]
    (from-long
     (+ (min start end)
        (long (Math/abs (- start end)))))))

(defn random-ratings
  "Generate random spot ratings."
  [db n & {:keys [start end]}]
  (let [start (or start (minus (now) (years 1)))
        end (or end (now))
        spots (map :id (run (select db [:id] (from :spots))))
        users (map :id (run (select db [:id] (from :users))))]
    (for [n (range 0 n)
          :let [time (to-date (rand-time start end))]]
      {:user-id (rand-nth users)
       :spot-id (rand-nth spots)
       :rating (inc (rand-int 5))
       :created-at time
       :updated-at time})))

(defn save-random-ratings
  "Generate random spot ratings and save them to the database."
  [db n & opts]
  (->> (apply random-ratings db n opts)
       (map #(save-rating db %1))
       (doall)))

(defn create-rating
  "Create a new rating."
  [db rating]
  (let [row (insert-rating db rating)]
    (rating-by-id db (:id row))))
