(ns burningswell.db.photos
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.images :as images]
            [burningswell.db.schemas :refer :all]
            [burningswell.db.util :refer :all]
            [burningswell.flickr :as flickr]
            [clojure.core :as core]
            [clojure.set :refer [rename-keys]]
            [datumbazo.core :as sql :refer :all
             :exclude [like delete insert update deftable]]
            [datumbazo.table :refer [deftable]]
            [geo.core :refer [point-x point-y point]]
            [hal.core :as hal]
            [schema.core :as s])
  (:import sqlingvo.db.Database))

(defn- row [photo]
  (assoc (select-keys photo [:location :url :status :title])
         :flickr-id (-> photo :flickr :id)
         :flickr-owner-id (-> photo :flickr :owner :id)
         :flickr-owner-name (-> photo :flickr :owner :name)
         :flickr-owner-url (-> photo :flickr :owner :url)
         :user-id (-> photo :_embedded :user :id)))

(defn- select-all [db & [opts]]
  (select db [:photos.id
              (as '(cast :photos.location :geometry) :location)
              :photos.url
              :photos.status
              :photos.title
              :photos.likes
              :photos.dislikes
              :photo-likes.like
              :photos.created-at
              :photos.updated-at
              (as `(json_build_object
                    "id" :photos.flickr-id
                    "owner"
                    (case (and (is-null :photos.flickr-owner-id)
                               (is-null :photos.flickr-owner-name)
                               (is-null :photos.flickr-owner-url)) nil
                          (json_build_object
                           "id" :photos.flickr-owner-id
                           "name" :photos.flickr-owner-name
                           "url" :photos.flickr-owner-url)))
                  :flickr)
              (as `(json_build_object
                    "user" (json-embed-user :users))
                  :_embedded)]
    (from :photos)
    (join :users '(on (= :users.id :photos.user-id)) :type :left)
    (join :photo-likes
          `(on (and (= :photo-likes.photo-id :photos.id)
                    (= :photo-likes.user-id ~(-> opts :user :id))))
          :type :left)
    (fulltext (:query opts) :photos.title)
    (order-by (or (:order-by opts)
                  (desc :photos.created-at)))
    (paginate (:page opts) (:per-page opts))))

(s/defn all :- [Photo]
  "Return all photos in `db`."
  [db :- Database & [opts]]
  @(select-all db opts))

(s/defn insert-photos-spots
  [db :- Database photo :- Photo spot :- Spot & [user]]
  (->> @(sql/insert db :photos-spots []
          (values [{:photo-id (:id photo)
                    :spot-id (:id spot)
                    :user-id (:id user)}]))
       first :count pos?))

(s/defn update-photos-spots
  [db :- Database photo :- Photo spot :- Spot & [user]]
  (->> @(sql/update db :photos-spots
          {:photo-id (:id photo)
           :spot-id (:id spot)
           :user-id (:id user)}
          (where `(and (= :photos-spots.photo-id ~(:id photo))
                       (= :photos-spots.spot-id ~(:id spot)))))
       first :count pos?))

(s/defn add-spot
  "Associate `photo` with the `spot`."
  [db :- Database photo :- Photo spot :- Spot & [user]]
  (assert (:id spot))
  (assert (:id photo))
  (or (update-photos-spots db photo spot user)
      (insert-photos-spots db photo spot user)))

(s/defn by-id :- (s/maybe Photo)
  "Return the country in `db` by `id`."
  [db :- Database id :- s/Num & [opts]]
  (first @(compose
           (select-all db opts)
           (where `(= :photos.id (cast ~id :integer))))))

(defn by-flickr-id
  "Return the photo by it's Flickr `id`."
  [db id & [opts]]
  (first @(compose
           (select-all db opts)
           (where `(= :photos.flickr-id (cast ~id :bigint))))))

(defn assoc-images [db photos]
  (let [images (images/by-photos db photos)
        images (images/group-by-photo-id images)]
    (map (fn [photo]
           (->> (get images (:id photo))
                (map images/select-embedded)
                (images/zip-by-label)
                (assoc photo :images)))
         photos)))

(s/defn by-countries
  "Return the cover photos of `countries` from `db`."
  [db :- Database countries :- [Country] & [opts]]
  (let [photo-ids (map (comp :id :photo :_embedded) countries)]
    (cond->> @(compose
               (select-all db opts)
               (where `(in :photos.id ~photo-ids)))
      (:images opts)
      (assoc-images db))))

(defn by-country
  "Return the photos of `country` from `db`."
  [db country & [opts]]
  (->> @(compose
         (select-all db opts)
         (join :photos-countries.photo-id :photos.id)
         (where `(= :photos-countries.country-id ~(:id country))))
       (:images opts)
       (assoc-images db)))

(s/defn by-spot
  "Return all photos of `spot` from `db`."
  [db :- Database spot :- Spot & [opts]]
  (cond->> @(compose
             (select-all db opts)
             (join :photos-spots.photo-id :photos.id)
             (where `(= :photos-spots.spot-id ~(:id spot))))
    (:images opts)
    (assoc-images db)))

(s/defn by-spots
  "Return the cover photos of `spots` from `db`."
  [db :- Database spots :- [Spot] & [opts]]
  (let [photo-ids (map (comp :id :photo :_embedded) spots)]
    (cond->> @(compose
               (select-all db opts)
               (where `(in :photos.id ~photo-ids)))
      (:images opts)
      (assoc-images db))))

(s/defn by-regions
  "Return the cover photos of `regions` from `db`."
  [db :- Database regions :- [Region] & [opts]]
  (let [photo-ids (map (comp :id :photo :_embedded) regions)]
    (cond->> @(compose
               (select-all db opts)
               (where `(in :photos.id ~photo-ids)))
      (:images opts)
      (assoc-images db))))

(defn by-region
  "Return the photos of `region` from `db`."
  [db region & [opts]]
  (->> @(compose
         (select-all db opts)
         (join :photos-regions.photo-id :photos.id)
         (where `(= :photos-regions.region-id ~(:id region))))
       (:images opts)
       (assoc-images db)))

(s/defn delete
  "Delete `photo` from `db`."
  [db :- Database photo :- Photo]
  (->> @(sql/delete db :photos
          (where `(= :photos.id
                     ~(:id photo))))
       first :count))

(defn delete-by-spot
  "Delete all the photos of `spot` in `db`."
  [db spot]
  @(sql/delete db :photos
     (sql/where `(in :id ~(sql/select db [:photo-id]
                            (sql/from :photos-spots)
                            (where `(= :photos-spots.spot-id ~(:id spot))))))))

(s/defn insert
  "Insert `photo` into `db`."
  [db :- Database photo]
  (->> @(sql/insert db :photos []
          (values [(row photo)])
          (returning :id))
       first :id (by-id db)))

(s/defn update
  "Update `photo` in `db`."
  [db :- Database photo]
  (->> @(sql/update db :photos
          (row photo)
          (where `(or (= :photos.id ~(:id photo))
                      (= :photos.flickr-id ~(-> photo :flickr :id))))
          (returning :id))
       first :id (by-id db)))

(s/defn save :- Photo
  "Save `photo` to `db`."
  [db :- Database photo]
  (or (update db photo)
      (insert db photo)))

(s/defn with-images :- [Photo]
  "Return all photos in `db` with their images."
  [db :- Database & [opts]]
  (let [photos (all db opts)]
    (assoc-images db photos)))

(s/defn with-images-by-id :- Photo
  "Return the photo in `db` by `id` with it's images."
  [db :- Database id :- s/Num & [opts]]
  (if-let [photo (by-id db id opts)]
    (first (assoc-images db [photo]))))

(s/defn like :- Photo
  "Like the photo in `db` as `user`."
  [db :- Database photo :- Photo user :- User]
  @(select db [`(upsert-photo-like ~(:id photo) ~(:id user) true)])
  (by-id db (:id photo) {:user user}))

(s/defn dislike :- Photo
  "Dislike the photo in `db` as `user`."
  [db :- Database photo :- Photo user :- User]
  @(select db [`(upsert-photo-like ~(:id photo) ~(:id user) false)])
  (by-id db (:id photo) {:user user}))

(s/defn likes :- s/Any
  "Return the likes and dislikes for `photo` in `db`."
  [db :- Database photo :- Photo & [opts]]
  @(select db [:*]
     (from :photo-likes)
     (where `(= :photo-likes.photo-id ~(:id photo)))
     (order-by (desc :photo-likes.updated-at))
     (paginate (:page opts) (:per-page opts))))
