(ns burningswell.db.user-locations
  (:require [burningswell.db.addresses :as addresses]
            [burningswell.db.util :refer :all]
            [clj-time.core :refer [before? now plus hours]]
            [clj-time.coerce :refer [to-date-time]]
            [datumbazo.core :as sql :refer [column table]]
            [datumbazo.table :refer [deftable]]
            [datumbazo.record :refer [select-class]]))

(deftable user-locations
  "The user locations datasources table."
  (column :id :serial :primary-key? true)
  (column :address-id :integer :references :addresses/id)
  (column :user-id :integer :not-null? true :references :users/id)
  (column :location :geography :not-null? true)
  (column :last-seen-at :timestamp :not-null? true)
  (column :created-at :timestamp :not-null? true)
  (column :updated-at :timestamp :not-null? true))

(defn columns
  "Return the user location columns."
  [& [opts]]
  [:user-locations.id
   :user-locations.address-id
   :user-locations.user-id
   (sql/as '(cast :user-locations.location :geometry) :location)
   :user-locations.last-seen-at
   :user-locations.updated-at
   :user-locations.created-at])

(defmethod select-class UserLocation
  [db class & [{:keys [distance location page per-page] :as opts}]]
  (sql/select db (columns opts)
    (sql/from :user-locations)
    (within-distance-to :user-locations.location location distance)
    (when location (order-by-distance :user-locations.location location))))

(defn save-all!
  "Save all `records` to `db`."
  [db records & [opts]]
  {:pre [(sql/db? db)]}
  @(sql/insert db :user-locations []
     (sql/values records)
     (sql/on-conflict [:user-id :location :last-seen-at]
       (sql/do-update {:updated-at :EXCLUDED.updated-at}))
     (apply sql/returning (columns opts))))

(defn by-user
  "Return all locations of `user` from `db`."
  [db user & [opts]]
  {:pre [(sql/db? db)]}
  @(sql/compose
    (select-class db UserLocation opts)
    (sql/where `(= :user-locations.user-id ~(:id user)) :and)
    (sql/order-by (sql/desc :last-seen-at))))

(defn last-location
  "Return the last location of `user` in `db`."
  [db user & [opts]]
  {:pre [(sql/db? db)]}
  (->> (assoc opts :page 1 :per-page 1)
       (by-user db user)
       (first)))

(defn within-time-range? [time-1 time-2]
  (let [time-1 (to-date-time time-1)
        time-2 (to-date-time time-2)]
    (before? time-2 (plus time-1 (hours 1)))))

(defn save-latest-location
  "Save the latest `location` at `last-seen-at` for `user` to `db`."
  ([db user location]
   (save-latest-location db user location (now)))
  ([db user location last-seen-at]
   (let [opts {:location location :distance 0.5}
         last-location (last-location db user opts)]
     (->> (if (and last-location
                   (within-time-range?
                    (:last-seen-at last-location) last-seen-at))
            (update! db (assoc last-location :last-seen-at last-seen-at))
            (insert! db {:address-id (:id (addresses/by-location db location))
                         :last-seen-at last-seen-at
                         :location location
                         :user-id (:id user)}))
          :id (by-id db)))))
