(ns burningswell.db.schemas
  (:require [schema.core :as s])
  (:import [org.postgis Point]))

(defn add-created-at
  "Assoc the :created-at timestamp to `schema`."
  [schema]
  (assoc schema :created-at s/Inst))

(defn add-updated-at
  "Assoc the :updated-at timestamp to `schema`."
  [schema]
  (assoc schema :updated-at s/Inst))

(defn add-timestamps
  "Assoc the :created-at and :updated-at timestamps to `schema`."
  [schema]
  (-> (add-created-at schema)
      (add-updated-at)))

(s/defschema GeoJSON
  "The schema for a GeoJSON data structure."
  {:coordinates [s/Any]
   :type s/Str
   s/Any s/Any})

(s/defschema EmbeddedTimeZone
  "The schema for an embedded time zone."
  {:id s/Int
   :offset s/Int
   :places [s/Str]})

(s/defschema TimeZone
  "The schema for a time zone."
  (-> (assoc EmbeddedTimeZone :natural-earth-id s/Int)
      (add-timestamps)))

(s/defschema FlickrUser
  "The schema for a Flickr user."
  {:id (s/maybe s/Str)
   :name (s/maybe s/Str)
   :url (s/maybe s/Str)})

(s/defschema EmbeddedPhoto
  "The schema for an embedded photo."
  {:flickr
   {:id (s/maybe s/Int)
    :owner (s/maybe FlickrUser)}
   :id s/Int
   :title s/Str
   s/Any s/Any})

(s/defschema Continent
  "The schema for a continent."
  (-> {:code s/Str
       :country-count s/Int
       :id s/Int
       :name s/Str
       s/Any s/Any}
      (add-timestamps)))

(s/defschema EmbeddedContinent
  "The schema for an embedded continent."
  (select-keys Continent [:id :name s/Any]))

(s/defschema Country
  "The schema for a country."
  (-> {:_embedded
       {:continent EmbeddedContinent
        :photo (s/maybe EmbeddedPhoto)
        s/Any s/Any}
       :airport-count s/Int
       :area (s/maybe s/Int)
       :fips-code (s/maybe s/Str)
       :id s/Int
       :iso-3166-1-alpha-2 s/Str
       :iso-3166-1-alpha-3 s/Str
       :iso-3166-1-numeric s/Int
       :location (s/maybe Point)
       :name s/Str
       :phone-prefix (s/maybe s/Str)
       :population s/Int
       :port-count s/Int
       :region-count s/Int
       :spot-count s/Int
       :user-count s/Int
       s/Any s/Any}
      (add-timestamps)))

(s/defschema EmbeddedCountry
  "The schema for an embedded country."
  (select-keys Country [:id :name :iso-3166-1-alpha-2 s/Any]))

;; Regions

(s/defschema Region
  "The schema for a region."
  (-> {:_embedded
       {:country EmbeddedCountry
        :photo (s/maybe EmbeddedPhoto)
        s/Any s/Any}
       :airport-count s/Int
       :id s/Int
       :name s/Str
       :location (s/maybe Point)
       :port-count s/Int
       :spot-count s/Int
       :user-count s/Int
       s/Any s/Any}
      (add-timestamps)))

(s/defschema EmbeddedRegion
  "The schema for an embedded region."
  (select-keys Region [:id :name s/Any]))

;; Users

(s/defschema User
  "The schema for a user"
  (-> {:_embedded
       {:country (s/maybe EmbeddedCountry)
        :region (s/maybe EmbeddedRegion)
        s/Any s/Any}
       :first-name (s/maybe s/Str)
       :id s/Int
       :last-name (s/maybe s/Str)
       :name (s/maybe s/Str)
       :username (s/maybe s/Str)
       (s/optional-key :email) (s/maybe s/Str)
       (s/optional-key :facebook-id) (s/maybe s/Str)
       (s/optional-key :facebook-url) (s/maybe s/Str)
       (s/optional-key :google-id) (s/maybe s/Str)
       (s/optional-key :google-url) (s/maybe s/Str)
       (s/optional-key :locale) (s/maybe s/Str)
       (s/optional-key :location) (s/maybe Point)
       (s/optional-key :twitter-id) (s/maybe s/Str)
       (s/optional-key :twitter-url) (s/maybe s/Str)
       s/Any s/Any}
      (add-timestamps)))

(s/defschema EmbeddedUser
  "The schema for an embedded user."
  (select-keys User [:id :username s/Any]))

;; Addresses

(s/defschema Address
  "The schema for an address."
  (-> {:_embedded
       {:country EmbeddedCountry
        :region (s/maybe EmbeddedRegion)
        :user (s/maybe EmbeddedUser)
        s/Any s/Any}
       :city (s/maybe s/Str)
       :formatted (s/maybe s/Str)
       :id s/Int
       :location Point
       :postal-code (s/maybe s/Str)
       :street-name (s/maybe s/Str)
       :street-number (s/maybe s/Str)
       s/Any s/Any}
      (add-timestamps)))

;; Airports

(s/defschema Airport
  "The schema for a airport."
  (-> {:_embedded
       {:country EmbeddedCountry
        :region (s/maybe EmbeddedRegion)
        s/Any s/Any}
       :gps-code s/Str
       :iata-code s/Str
       :id s/Int
       :location Point
       :name s/Str
       :wikipedia-url (s/maybe s/Str)
       s/Any s/Any}
      (add-timestamps)))

;; Roles

(s/defschema Role
  "The schema for a role."
  (-> {:description s/Str
       :name s/Str
       :id s/Int
       s/Any s/Any}
      (add-timestamps)))

;; Spots

(s/defschema Spot
  "The schema for a spot."
  (-> {:_embedded
       {:country (s/maybe EmbeddedCountry)
        :region (s/maybe EmbeddedRegion)
        :user (s/maybe EmbeddedUser)
        :photo (s/maybe EmbeddedPhoto)
        :time-zone (s/maybe EmbeddedTimeZone)
        s/Any s/Any}
       :id s/Int
       :location Point
       :name s/Str
       :visible s/Any
       s/Any s/Any}
      (add-timestamps)))

(s/defschema EmbeddedSpot
  "The schema for an embedded spot."
  (select-keys Spot [:id :name s/Any]))

;; Comments

(s/defschema Comment
  "The schema for a spot."
  (-> {:_embedded
       {:spot EmbeddedSpot
        :user EmbeddedUser
        s/Any s/Any}
       :id s/Int
       :content s/Str
       :visible s/Str
       s/Any s/Any}
      (add-timestamps)))

;; Images

(s/defschema Image
  "The schema for a photo."
  (-> {:content-length (s/maybe s/Int)
       :content-md5 (s/maybe s/Str)
       :content-type (s/maybe s/Str)
       :height (s/maybe s/Int)
       :id s/Int
       :label s/Str
       :url (s/maybe s/Str)
       :storage-key (s/maybe s/Str)
       :width (s/maybe s/Int)
       s/Any s/Any}
      (add-timestamps)))

(s/defschema EmbeddedImage
  "The schema for an embedded image"
  (dissoc Image :content-md5 :storage-key))

(s/defschema Photo
  "The schema for a photo."
  (-> {:_embedded
       {:user (s/maybe EmbeddedUser)
        s/Any s/Any}
       :flickr
       {:id (s/maybe s/Int)
        :owner (s/maybe FlickrUser)
        s/Any s/Any}
       :id s/Int
       :location (s/maybe Point)
       :url (s/maybe s/Str)
       :status (s/maybe s/Str)
       :title s/Str
       :like (s/maybe s/Bool)
       :likes s/Int
       :dislikes s/Int
       (s/optional-key :images) {s/Keyword EmbeddedImage}
       s/Any s/Any}
      (add-timestamps)))

(s/defschema Port
  "The schema for a port."
  (-> {:_embedded
       {:country EmbeddedCountry
        :region (s/maybe EmbeddedRegion)
        s/Any s/Any}
       :id s/Int
       :location Point
       :name s/Str
       :type s/Str
       :website-url (s/maybe s/Str)
       s/Any s/Any}
      (add-timestamps)))

(s/defschema Rating
  "The schema for a rating."
  (-> {:id s/Int
       :rating s/Int
       :spot-id s/Int
       :user-id s/Int
       :rated-at s/Inst
       s/Any s/Any}
      (add-timestamps)))

(s/defschema Session
  "The schema for a session."
  (-> {:id s/Int
       :spot-id s/Int
       :user-id s/Int
       (s/optional-key :started-at) s/Inst
       (s/optional-key :stopped-at) s/Inst
       :rating s/Int
       s/Any s/Any}
      (add-timestamps)))

(s/defschema WeatherEntry
  "The schema for the weather at a surf spot."
  {:unit s/Str
   :value s/Num})

(s/defschema PrimaryWaveDirection
  "The schema for the primary wave direction."
  {(s/optional-key :dirpwsfc) WeatherEntry})

(s/defschema SecondaryWaveDirection
  "The schema for the secondary wave direction."
  {(s/optional-key :dirswsfc) WeatherEntry})

(s/defschema SignificantWaveHeight
  "The schema for the significant wave height."
  {(s/optional-key :htsgwsfc) WeatherEntry})

(s/defschema PrimaryWaveMeanPeriod
  "The schema for the primary wave mean period"
  {(s/optional-key :perpwsfc) WeatherEntry})

(s/defschema SecondaryWaveMeanPeriod
  "The schema for the secondary wave mean period."
  {(s/optional-key :perswsfc) WeatherEntry})

(s/defschema TotalCloudCover
  "The schema for the total cloud cover."
  {(s/optional-key :tcdcclm) WeatherEntry})

(s/defschema SurfaceTemperature
  "The schema for the surface temperature."
  {(s/optional-key :tmpsfc) WeatherEntry})

(s/defschema WindUCmponent
  "The schema for the U wind component."
  {(s/optional-key :ugrdsfc) WeatherEntry})

(s/defschema WindVComponent
  "The schema for the V wind component."
  {(s/optional-key :vgrdsfc) WeatherEntry})

(s/defschema WindDirection
  "The schema for the wind direction."
  {(s/optional-key :wdirsfc) WeatherEntry})

(s/defschema WindSpeed
  "The schema for the wind speed."
  {(s/optional-key :windsfc) WeatherEntry})

(s/defschema WindWaveDirection
  "The schema for the wind wave direction."
  {(s/optional-key :wvdirsfc) WeatherEntry})

(s/defschema WindWaveMeanPeriod
  "The schema for mean period of wind waves."
  {(s/optional-key :wvpersfc) WeatherEntry})

(s/defschema Weather
  "The schema for the weather at a surf spot."
  {s/Inst
   (merge PrimaryWaveDirection
          PrimaryWaveMeanPeriod
          SecondaryWaveDirection
          SecondaryWaveMeanPeriod
          SignificantWaveHeight
          SurfaceTemperature
          TotalCloudCover
          WindDirection
          WindSpeed
          WindUCmponent
          WindVComponent
          WindWaveDirection
          WindWaveMeanPeriod)})

(s/defschema WeatherModel
  "The schema for a weather model."
  (-> {:description s/Str
       :dods s/Str
       :id s/Int
       :latest-reference-time (s/maybe s/Inst)
       :name s/Str
       :pattern (s/maybe s/Str)
       :res-x s/Num
       :res-y s/Num
       s/Any s/Any}
      (add-timestamps)))

(s/defschema EmbeddedWeatherModel
  "The schema for an embedded weather model."
  (select-keys WeatherModel [:id :name s/Any]))

(s/defschema WeatherVariable
  "The schema for a weather variable."
  (-> {:id s/Int
       :name s/Str
       :description s/Str
       :unit s/Str
       s/Any s/Any}
      (add-timestamps)))

(s/defschema EmbeddedWeatherVariable
  "The schema for an embedded weather variable."
  (select-keys WeatherVariable [:id :name s/Any]))

(s/defschema WeatherDataset
  "The schema for a weather dataset."
  (-> {:_embedded
       {:model EmbeddedWeatherModel
        :variable EmbeddedWeatherModel
        s/Any s/Any}
       :das s/Str
       :dds s/Str
       :dods s/Str
       :id s/Int
       :reference-time s/Inst
       :valid-time s/Inst
       s/Any s/Any}
      (add-timestamps)))

(s/defschema WeatherDatasource
  "The schema for a weather datasource."
  (-> {:_embedded
       {:model WeatherModel
        s/Any s/Any}
       :id s/Int
       s/Any s/Any}
      (add-timestamps)))

(s/defschema WeatherForecast
  "The schema for a weather forecast."
  (-> {:id s/Int
       :status s/Any
       s/Any s/Any}
      (add-timestamps)))
