(ns burningswell.db.roles
  (:refer-clojure :exclude [distinct group-by update])
  (:require [burningswell.db.util :refer :all]
            [datumbazo.core :refer :all :as sql
             :exclude [delete insert update]]))

(derive ::admin ::surfer)
(derive ::editor ::surfer)

(defn role-keyword [role]
  (keyword (str "burningswell.db.roles/" (name role))))

(defn- select-all [db & [opts]]
  (let [{:keys [page per-page query]} opts]
    (select db [:roles.id
                :roles.name
                :roles.description
                :roles.created-at
                :roles.updated-at]
      (from :roles)
      (fulltext query :roles.name :roles.description)
      (paginate page per-page))))

(defn all
  "Returns all users in `db`.."
  [db & [opts]]
  @(select-all db opts))

(defn by-id
  "Find the role by `id` in `db`."
  [db id & [opts]]
  (first @(compose
           (select-all db opts)
           (where `(= :roles.id (cast ~id :int))))))

(defn by-name
  "Find the role by `name` in `db`."
  [db name & [opts]]
  (first @(compose
           (select-all db opts)
           (where `(= :roles.name (cast ~name :citext))))))

(defn by-user
  "Find all roles of `user` in `db`."
  [db user & [opts]]
  @(compose
    (select-all db opts)
    (join :roles-users.role-id :roles.id)
    (where `(= :roles-users.user-id ~(:id user)))))

(defn delete
  "Delete `role` from `db`."
  [db role]
  (->> @(sql/delete db :roles
                    (where `(= :roles.id
                               ~(:id role))))
       first :count))

(defn insert
  "Insert `user` into `db`."
  [db role]
  (->> @(sql/insert db :roles []
                    (values [role])
                    (returning :id))
       first :id (by-id db)))

(defn update
  "Update `role` in `db`."
  [db role]
  (->> @(sql/update db :roles role
                    (where `(= :roles.id ~(:id role)))
                    (returning :id))
       first :id (by-id db)))

(defn admin
  "Returns the admin role."
  [db]
  (by-id db 1))

(defn editor
  "Returns the editor role."
  [db]
  (by-id db 2))

(defn surfer
  "Returns the surfer role."
  [db]
  (by-id db 3))
