(ns burningswell.http.headers
  (:require [clojure.string :refer [lower-case split]]
            [clojure.test :refer [is]]
            [burningswell.number :refer :all]))

(defn accept-entry-compare-key [accept-entry]
  (parse-double (get (:params accept-entry) "q" "1") :junk-allowed true))

(defn parse-accept-param [param]
  (apply hash-map (split param #"\s*=\s*")))

(defrecord AcceptEntry [content-type params]
  Comparable
  (compareTo [this other]
    (compare
     (accept-entry-compare-key this)
     (accept-entry-compare-key other))))

(defn parse-accept-entry [entry]
  (let [result (split entry #"\s*;\s*")]
    (AcceptEntry.
     (first result)
     (apply merge (map parse-accept-param (rest result))))))

(defn parse-accept-header [text]
  (if text (sort (map parse-accept-entry (split text #"\s*,\s*")))))

(defn headers
  "Returns the lower-cased headers of the map."
  [m]
  (reduce
   #(assoc %1 (lower-case (name (first %2))) (second %2))
   {} (:headers m)))

(defn int-header
  "Returns the header of the map as int."
  [m header-name]
  (parse-integer (get (headers m) header-name) :junk-allowed true))

(defn content-type
  "Returns the Content Type header from map."
  [m] (get (headers m) "content-type"))

(defn location-header
  "Returns the Location header from map."
  [m] (get (headers m) "location"))

(defn clojure?
  "Returns true if the map's Content-Type header is set to
  \"application/clojure\"."
  [m] (= "application/clojure" (content-type m)))

(defn json?
  "Returns true if the map's Content-Type header is set to
  \"application/json\"."
  [m] (= "application/json" (content-type m)))

(defn pagination-headers [coll]
  (let [{:keys [page per-page total]} (meta coll)]
    (if page
      {"x-pagination-page" (str page)
       "x-pagination-per-page" (str per-page)
       "x-pagination-total" (str total)})))

(defn x-pagination-page
  "Return the X-Pagination-Page header from map."
  [m] (int-header m "x-pagination-page"))

(defn x-pagination-per-page
  "Return the X-Pagination-Per-Page header from map."
  [m] (int-header m "x-pagination-per-page"))

(defn x-pagination-total
  "Return the X-Pagination-Total header from map."
  [m] (int-header m "x-pagination-total"))

(defn has-x-pagination-headers
  "Verify that response has the x-pagination headers."
  [response page per-page total]
  (is (= page (x-pagination-page response)))
  (is (= per-page (x-pagination-per-page response)))
  (is (= total (x-pagination-total response))))
