(ns burningswell.json
  (:import [com.fasterxml.jackson.core JsonGenerator]
           [java.io File PrintWriter])
  (:refer-clojure :exclude [read-string])
  (:require [burningswell.time :refer :all]
            [cheshire.core :refer [generate-string]]
            [cheshire.generate :refer [JSONable to-json]]
            [clojure.data.json :as json]
            [clojure.data.json :refer [-write JSONWriter]]
            [clojure.string :refer [blank?]]))

(defn json-str [s]
  (json/write-str s))

(defn read-json [s]
  (cond
   (and (string? s) (not (blank? s)))
   (json/read-str s :key-fn keyword)
   (instance? java.io.InputStream s)
   (let [s (slurp s)]
     (if-not (blank? s)
       (json/read-str s :key-fn keyword)))))

(defn read-string [s]
  (json/read-str s :key-fn keyword))

(defn- make-point [^org.postgis.Point point]
  {:type "Point" :coordinates [(.getX point) (.getY point) (.getZ point)]})

(defn- write-time [time ^PrintWriter out]
  (-write (format-time time) out))

(extend-type java.lang.Class
  JSONWriter
  (-write [class out]
    (-write (.getName class) out)))

(extend-type java.util.Date
  JSONWriter
  (-write [date out]
    (-write (format-time date) out)))

(extend-type java.sql.Timestamp
  JSONWriter
  (-write [timestamp out]
    (write-time timestamp out)))

(extend-type org.joda.time.DateTime
  JSONable
  (to-json [date-time ^JsonGenerator generator]
    (.writeString generator (format-time date-time)))
  JSONWriter
  (-write [date-time out]
    (write-time date-time out)))

(extend-type org.postgis.PGgeometry
  JSONable
  (to-json [geom generator]
    (to-json (.getGeometry geom) generator))
  JSONWriter
  (-write [geom out]
    (-write (.getGeometry geom) out)))

(extend-type org.postgis.Point
  JSONable
  (to-json [point generator]
    (.writeRaw generator (generate-string (make-point point))))
  JSONWriter
  (-write [point out]
    (-write (make-point point) out)))

(defn wrap-json-params
  "Parse the request params as JSON."
  [handler & [default]]
  (fn [{:keys [content-type headers body params] :as request}]
    (handler
     (if (re-find #"(?i)application/json"
                  (or content-type (get headers "content-type" "")))
       (let [body (read-json body)]
         (assoc request :body body :params (merge params body)))
       request))))

(defn wrap-json-response
  "Encode the response as JSON."
  [handler & [default]]
  (let [encode
        (fn [response]
          (-> (update-in response [:body] json-str)
              (assoc-in [:headers "content-type"] "application/json")))]
    (fn [{:keys [headers params] :as request}]
      (let [response (handler request)
            accept (get headers "accept" "")]
        (if (or (re-find #"(?i)application/json" accept) default)
          (cond
           (nil? response)
           response
           (instance? File response)
           response
           (or (:body response) (integer? (:status response)))
           (encode (merge {:status 200} response))
           :else (encode {:status 200 :body response}))
          response)))))
