(ns burningswell.transit
  (:require [clj-http.client]
            [clj-time.coerce :refer [to-long]]
            [cognitect.transit :as transit]
            [geo.transit :as geo-transit]
            [schema.utils])
  (:import [java.io ByteArrayInputStream ByteArrayOutputStream]
           [org.joda.time DateTime]
           [schema.utils ValidationError]))

(declare encode decode)

(defn coerce-transit-body
  [request {:keys [body] :as resp}]
  (assoc resp :body (decode (slurp body))))

(defmethod clj-http.client/coerce-content-type :application/transit+json
  [req resp]
  (coerce-transit-body req resp))

(defn wrap-transit-form-params
  "Middleware wrapping the submission of form parameters in the transit fromat."
  [client]
  (fn [{:keys [form-params content-type method request-method]
        :or {content-type :x-www-form-urlencoded}
        :as req}]
    (if (and form-params
             (#{:post :put :patch} (or request-method method))
             (#{:transit
                :application/transit+json
                "application/transit+json"}
              content-type))
      (client (-> req
                  (dissoc :form-params)
                  (assoc :content-type "application/transit+json"
                         :body (encode form-params))))
      (client req))))

(def read-validation-error
  "Read a validation error."
  (transit/read-handler
   (fn [[srid coordinates]]
     nil)))

(def write-class
  "Write a Class as a symbol"
  (transit/write-handler
   (constantly "$")
   (fn [class] (str class))
   (constantly nil)))

(def write-date-time
  "Write org.joda.time.DateTime."
  (transit/write-handler
   (constantly "m")
   (fn [date-time]
     (str (to-long date-time)))
   (constantly nil)))

(def write-validation-error
  "Write a validation error."
  (transit/write-handler
   (constantly "burningswell/validation-error")
   (fn [error]
     (schema.utils/validation-error-explain error))
   (constantly nil)))

(def read-handlers
  "The Transit read handlers."
  (merge
   geo-transit/read-handlers
   {"burningswell/validation-error" read-validation-error}))

(def write-handlers
  "The Transit write handlers."
  (merge
   geo-transit/write-handlers
   {Class write-class
    DateTime write-date-time
    ValidationError write-validation-error}))

(defn encode
  "Encode `payload` into Transit format."
  [payload & [opts]]
  (let [format (or (:format opts) :json)
        stream (ByteArrayOutputStream.)
        writer (transit/writer stream format {:handlers write-handlers})
        _ (transit/write writer payload)]
    (case format
      :json (.toString stream)
      :msgpack (.toByteArray stream))))

(defn decode
  "Decode `payload` from Transit format."
  [payload & [opts]]
  (let [format (or (:format opts) :json)
        stream (ByteArrayInputStream.
                (case format
                  :json (.getBytes payload)
                  :msgpack payload))
        reader (transit/reader stream format {:handlers read-handlers})]
    (transit/read reader)))
