(ns burningswell.system
  (:require [com.stuartsierra.component :as component]))

(defn on-shutdown
  "Run the function `f` on JVM shutdown."
  [f]
  (.addShutdownHook (Runtime/getRuntime) (Thread. f)))

(defmacro with-system
  "Start the `component`, bind the started component to
  `component-sym`, evaluate `body` and stop `component` again."
  [[component-sym component] & body]
  `(let [component# (component/start ~component)
         ~component-sym component#]
     (try ~@body
          (finally (component/stop component#)))))

(defn- define-system
  "Return a form to define a system constructor."
  [system-name args body]
  `(defn ~system-name ~args
     ~@body))

(defn- define-with-system
  "Return a form to define a macro that starts the system, evals the
  macro body and stops the system again."
  [system-name args]
  (let [args (map (fn [_] (gensym "arg")) args)]
    `(defmacro ~(symbol (str "with-" system-name))
       [[~'system-sym ~@args] ~'& ~'body]
       `(with-system [~~'system-sym (~~system-name ~~@args)]
          ~@~'body))))

(defmacro defsystem
  "Define a new system and helper functions."
  [system-name doc args & body]
  `(do ~(define-system system-name args body)
       ~(define-with-system system-name args)))
