(ns burningswell.transit
  (:require [clj-http.client]
            [cognitect.transit :as transit]
            [geo.transit :as geo-transit])
  (:import (java.io ByteArrayInputStream ByteArrayOutputStream)))

(def write-class-as-symbol
  "Write a Class as a symbol"
  (transit/write-handler
   (constantly "$")
   (fn [class] (str class))
   (constantly nil)))

(def read-handlers
  geo-transit/read-handlers)

(def write-handlers
  (merge geo-transit/write-handlers
         {Class write-class-as-symbol}))

(defn write-str
  "Return the data structure `x`, as a transit encoded string."
  [x]
  (let [baos (ByteArrayOutputStream.)
        writer (transit/writer baos :json {:handlers write-handlers})
        _ (transit/write writer x)
        ret (.toString baos)]
    (.reset baos)
    ret))

(defn read-str
  "Read a transit encoded data structure from `s`."
  [s]
  (let [bais (ByteArrayInputStream. (.getBytes s))
        reader (transit/reader bais :json {:handlers read-handlers})]
    (transit/read reader)))

(defn coerce-transit-body
  [request {:keys [body] :as resp}]
  (assoc resp :body (read-str (slurp body))))

(defmethod clj-http.client/coerce-content-type :application/transit+json
  [req resp]
  (coerce-transit-body req resp))

(defn wrap-transit-form-params
  "Middleware wrapping the submission of form parameters in the transit fromat."
  [client]
  (fn [{:keys [form-params content-type method request-method]
        :or {content-type :x-www-form-urlencoded}
        :as req}]
    (if (and form-params
             (#{:post :put :patch} (or request-method method))
             (#{:transit
                :application/transit+json
                "application/transit+json"}
              content-type))
      (client (-> req
                  (dissoc :form-params)
                  (assoc :content-type "application/transit+json"
                         :body (write-str form-params))))
      (client req))))
