(ns burningswell.time
  (:import java.io.Writer
           java.sql.Timestamp
           java.util.Date
           [org.joda.time DateTime DateTimeZone Interval]
           [org.joda.time.format DateTimeFormat DateTimeFormatter
            ISODateTimeFormat PeriodFormat])
  (:require [clj-time.format :refer [formatters formatter parse unparse]]
            [clj-time.core :refer [date-time interval now]]
            [clojure.instant :as inst]))

(def ^:dynamic *date-formatter* :date)
(def ^:dynamic *time-formatter* :date-time-no-ms)

(defn- format-with-local-zone
  [^DateTime date-time ^DateTimeFormatter formatter]
  (-> formatter
    (.withZone (DateTimeZone/forID "Europe/Berlin"))
    (.print date-time)))

(defn- parse-date-time [string formatter]
  (parse (formatters formatter) string))

(defprotocol TimeProtocol
  (to-ms [this] "Returns the time in milliseconds."))

(extend-type java.lang.Number
  TimeProtocol
  (to-ms [number] number))

(extend-type java.lang.String
  TimeProtocol
  (to-ms
    [string]
    (try
      (to-ms (parse-date-time string *time-formatter*))
      (catch IllegalArgumentException _
        (to-ms (parse-date-time string *date-formatter*))))))

(extend-type java.util.Calendar
  TimeProtocol
  (to-ms [calendar] (to-ms (.getTime calendar))))

(extend-type java.util.Date
  TimeProtocol
  (to-ms [date] (.getTime date)))

(extend-type org.joda.time.DateTime
  TimeProtocol
  (to-ms [date-time] (.getMillis date-time)))

(extend-type nil
  TimeProtocol
  (to-ms [_] nil))

(defn date-time?
  "Returns true if arg is a DateTime object, otherwise false."
  [arg] (isa? (class arg) DateTime))

(defn to-calendar
  "Convert the object to a java.util.Calendar."
  [object]
  (doto (java.util.Calendar/getInstance)
    (.setTimeInMillis (to-ms object))))

(defn to-date
  "Convert the object to a java.sql.Date."
  [object] (if object (java.sql.Date. (to-ms object))))

(defn to-date-time
  "Convert the object to a org.joda.time.DateTime."
  [object] (if object (DateTime. (long (to-ms object)) (DateTimeZone/UTC))))

(defn date-hour-path
  "Returns the date and hour as a path fragment."
  [date-time] (unparse (formatter "yyyy/MM/dd/HH") (to-date-time date-time)))

(defn format-date
  "Format the object with the default or the given date formatter."
  [object & [formatter]]
  (if object
    (unparse (or formatter (formatters *date-formatter*))
             (to-date-time object))))

(defn format-day-of-week
  "Format the day of week."
  [time & {:keys [short]}]
  (.print
   ^DateTimeFormat
   (if short
     (DateTimeFormat/forPattern "E")
     (DateTimeFormat/forPattern "EEEE"))
   ^DateTime
   (to-date-time time)))

(defn ^String format-interval
  "Format the interval using the default period formatter."
  [^Interval interval]
  (.print (PeriodFormat/getDefault) (.toPeriod interval)))

(defn ^String format-time
  "Format the object with the default or the given time formatter."
  [object & [formatter]]
  (if object
    (unparse (or formatter (formatters *time-formatter*))
             (to-date-time object))))

(defn sql-timestamp
  "Convert the object to a java.sql.Timestamp."
  [object] (if object (Timestamp. (to-ms object))))

(defn sql-timestamp?
  "Returns true if arg is a java.sql.Timestamp, otherwise false."
  [arg] (isa? (class arg) Timestamp))

(defn sql-timestamp-now
  "Returns the current time as java.sql.Timestamp."
  [] (sql-timestamp (now)))

(defmethod print-method DateTime
  [^DateTime t, ^Writer w]
  (print-method (Date. (.getMillis t)) w))

(defmethod print-dup DateTime
  [^DateTime t, ^Writer w]
  (print-dup (Date. (.getMillis t)) w))

(defn- construct-date-time
  [years months days hours minutes seconds nanoseconds
   offset-sign offset-hours offset-minutes]
  (-> (.getTimeInMillis
       ^java.util.Calendar
       (#'inst/construct-calendar
        years months days
        hours minutes seconds nanoseconds
        offset-sign offset-hours offset-minutes))
    (DateTime.) (.withZone DateTimeZone/UTC)))

(def read-instant-date-time
  "To read an instant as an DateTime, bind *data-readers* to a map
  with this var as the value for the 'inst key."
  (partial inst/parse-timestamp (inst/validated construct-date-time)))
