(ns burningswell.config.schema
  "The Burning Swell configuration schemas."
  (:require [schema.core :as s #?@(:cljs [:include-macros true])]))

(s/defschema Scheme
  "The schema for a protocol scheme."
  (s/enum :http :https))

(s/defschema Server
  "The schema for a server."
  {:bind-address s/Str
   :bind-port s/Int})

(s/defschema ApiClient
  "The schema for the API client."
  {:scheme Scheme
   :server-name s/Str
   :server-port (s/maybe s/Int)})

(s/defschema Aws
  "The schema for the AWS service."
  {:access-key s/Str
   :account-number s/Str
   :secret-key s/Str})

(s/defschema Figwheel
  "The schema for the Figwheel service."
  {:enable s/Bool})

(s/defschema ImageProxy
  "The schema for the Image Proxy service."
  {:backend-url s/Str
   :bind-address s/Str
   :bind-port s/Int
   :scheme Scheme
   :server-name s/Str
   :server-port (s/maybe s/Int)})

(s/defschema Broker
  "The schema for the broker service."
  {:password s/Str
   :scheme (s/enum :amqp)
   :server-name s/Str
   :server-port (s/maybe s/Int)
   :username s/Str
   :vhost s/Str})

(s/defschema Datomic
  "The schema for the Datomic config."
  {:url s/Str})

(s/defschema Db
  "The schema for the database service."
  {:name s/Str
   :password s/Str
   :scheme (s/enum :postgresql)
   :server-name s/Str
   :server-port (s/maybe s/Int)
   :username s/Str
   (s/optional-key :test?) s/Bool})

(s/defschema Facebook
  "The schema for the Facebook service."
  {:client-id s/Str
   :client-secret s/Str
   :redirect-uri s/Str})

(s/defschema Flickr
  "The schema for the Flickr service."
  {:api-key s/Str
   :client-id s/Str
   :client-secret s/Str})

(s/defschema GoogleMaps
  "The schema for the Google Maps service."
  {:api-key s/Str})

(s/defschema GoogleStorage
  "The schema for the Google Storage service."
  {:bucket s/Str})

(s/defschema Google
  "The schema for Google services."
  {:maps GoogleMaps
   :storage GoogleStorage})

(s/defschema Mapbox
  "The schema for the Mapbox service."
  {:api-key s/Str
   :style s/Str})

(s/defschema JWT
  "The schema for the JWT config."
  {:secret s/Str
   :expires-in s/Int})

(s/defschema Reload
  "The schema for the namespace reload service."
  {:disabled s/Bool
   :source-paths [s/Str]})

(s/defschema S3Backups
  "The schema for the S3 backup configuration."
  {:bucket s/Str
   :prefix s/Str})

(s/defschema S3Photos
  "The schema for the S3 photo configuration."
  {:bucket s/Str
   :prefix s/Str})

(s/defschema S3
  "The schema for the S3 configuration."
  {:backups S3Backups
   :photos S3Photos})

(s/defschema WebClient
  "The schema for the web client."
  {:scheme Scheme
   :server-name s/Str
   :server-port (s/maybe s/Int)})

(s/defschema Web
  "The schema for the web service."
  {:api-client ApiClient
   :async-js s/Bool
   :bind-address s/Str
   :bind-port s/Int
   :google Google
   :facebook Facebook
   :figwheel Figwheel
   :flickr Flickr
   :mapbox Mapbox
   :elements
   {:config s/Str
    :root s/Str
    :state s/Str}
   :ssl-redirect? s/Bool
   :web-client WebClient})

(s/defschema Api
  "The schema for the API service."
  {:api-client ApiClient
   :bind-address s/Str
   :bind-port s/Int
   :broker Broker
   :db Db
   :google Google
   :facebook Facebook
   :flickr Flickr
   :jwt JWT
   :reload Reload
   :ssl-redirect? s/Bool
   :web WebClient})

(s/defschema Scheduler
  "The schema for the scheduler service."
  {:broker Broker
   :db Db})

(s/defschema Weather
  "The schema for the weather service."
  {:server Server
   :broker Broker
   :db Db})

(s/defschema Worker
  "The schema for a worker."
  {:bind-address s/Str
   :bind-port s/Int})

(s/defschema Config
  "The schema for the config."
  {:api Api
   :api-client ApiClient
   :aws Aws
   :broker Broker
   :db Db
   :facebook Facebook
   :flickr Flickr
   :google Google
   :reload Reload
   :weather Weather
   :worker Worker
   :web Web
   :web-client WebClient})
