(ns burningswell.config.schema
  (:require [schema.core :as s]))

(def ApiClient
  "The schema for the API client."
  {:scheme (s/enum :http :https)
   :server-name s/Str
   :server-port (s/maybe s/Int)})

(def Aws
  "The schema for the AWS service."
  {:access-key s/Str
   :account-number s/Str
   :secret-key s/Str})

(def Figwheel
  "The schema for the Figwheel service."
  {:enable s/Bool})

(def ImageProxy
  "The schema for the Image Proxy service."
  {:backend-url s/Str
   :bind-address s/Str
   :bind-port s/Int
   :scheme (s/enum :http :https)
   :server-name s/Str
   :server-port (s/maybe s/Int)})

(def Broker
  "The schema for the broker service."
  {:password s/Str
   :scheme (s/enum :amqp)
   :server-name s/Str
   :server-port (s/maybe s/Int)
   :username s/Str
   :vhost s/Str})

(def Db
  "The schema for the database service."
  {:name s/Str
   :password s/Str
   :scheme (s/enum :postgresql)
   :server-name s/Str
   :server-port (s/maybe s/Int)
   :user s/Str
   (s/optional-key :test) s/Bool})

(def Facebook
  "The schema for the Facebook service."
  {:client-id s/Str
   :client-secret s/Str
   :redirect-uri s/Str})

(def Flickr
  "The schema for the Flickr service."
  {:api-key s/Str
   :client-id s/Str
   :client-secret s/Str})

(def GoogleMaps
  "The schema for the Google Maps service."
  {:api-key s/Str})

(def Google
  "The schema for Google services."
  {:maps GoogleMaps})

(def JWS
  "The schema for the JWS config."
  {:secret s/Str
   :expires-in s/Int})

(def Reload
  "The schema for the namespace reload service."
  {:disabled s/Bool
   :source-paths [s/Str]})

(def S3Backups
  "The schema for the S3 backup configuration."
  {:bucket s/Str
   :prefix s/Str})

(def S3Photos
  "The schema for the S3 photo configuration."
  {:bucket s/Str
   :prefix s/Str})

(def S3
  "The schema for the S3 configuration."
  {:backups S3Backups
   :photos S3Photos})

(def Web
  "The schema for the web service."
  {:api-client ApiClient
   :bind-address s/Str
   :bind-port s/Int
   :google Google
   :facebook Facebook
   :figwheel Figwheel
   :flickr Flickr
   :ssl-redirect? s/Bool})

(def WebClient
  "The schema for the web client."
  {:scheme (s/enum :http :https)
   :server-name s/Str
   :server-port (s/maybe s/Int)})

(def Api
  "The schema for the API service."
  {:api-client ApiClient
   :bind-address s/Str
   :bind-port s/Int
   :broker Broker
   :db Db
   :google Google
   :facebook Facebook
   :flickr Flickr
   :jws JWS
   :reload Reload
   :ssl-redirect? s/Bool
   :web WebClient})

(def Scheduler
  "The schema for the scheduler service."
  {:broker Broker
   :db Db})

(def Config
  "The schema for the config."
  {:api Api
   :api-client ApiClient
   :aws Aws
   :broker Broker
   :db Db
   :facebook Facebook
   :flickr Flickr
   :google Google
   :reload Reload
   :web Web
   :web-client WebClient})
