(ns burningswell.config.core
  (:require [burningswell.config.schema :refer :all]
            [clojure.edn :as edn]
            [schema.core :as s]))

(defn- bind-port
  "Return the bind port in `env` under `key`, or :port."
  [env key]
  (or (edn/read-string (get env key))
      (edn/read-string (:port env))))

(s/defn ^:always-validate api-client :- ApiClient
  "Return the API client config from `env`."
  [env]
  {:scheme (keyword (:bs-api-scheme env :https))
   :server-name (:bs-api-server-name env)
   :server-port (edn/read-string (:bs-api-server-port env))})

(s/defn ^:always-validate aws :- Aws
  "Return the AWS config from `env`."
  [env]
  {:access-key (:bs-aws-access-key env)
   :account-number (:bs-aws-account-number env)
   :secret-key (:bs-aws-secret-key env)})

(s/defn ^:always-validate figwheel :- Figwheel
  "Return the Figwheel config from `env`."
  [env]
  {:enable (or (edn/read-string (:bs-figwheel-enable env)) false)})

(s/defn ^:always-validate image-proxy :- ImageProxy
  "Return the Image Proxy config from `env`."
  [env]
  {:backend-url (:bs-image-proxy-backend-url env)
   :bind-address (:bs-image-proxy-bind-address env)
   :bind-port (or (bind-port env  :bs-image-proxy-bind-port) 20002)
   :scheme (keyword (:bs-image-proxy-scheme env :https))
   :server-name (:bs-image-proxy-server-name env)
   :server-port (edn/read-string (:bs-image-proxy-server-port env))})

(s/defn ^:always-validate broker :- Broker
  "Return the broker config from `env`."
  [env]
  {:password (:bs-broker-password env)
   :scheme (keyword (:bs-broker-scheme env :amqp))
   :server-name (:bs-broker-server-name env)
   :server-port (edn/read-string (:bs-broker-server-port env))
   :username (:bs-broker-username env)
   :vhost (or (:bs-broker-vhost env) "/")})

(s/defn ^:always-validate db :- Db
  "Return the database config from `env`."
  [env]
  {:name (:bs-db-name env)
   :password (:bs-db-password env)
   :test (true? (edn/read-string (:bs-db-rollback env)))
   :scheme (keyword (:bs-db-scheme env :postgresql))
   :server-name (:bs-db-server-name env)
   :server-port (edn/read-string (:bs-db-server-port env))
   :user (:bs-db-user env)})

(s/defn ^:always-validate facebook :- Facebook
  "Return the Facebook config from `env`."
  [env]
  {:client-id (:bs-facebook-client-id env)
   :client-secret (:bs-facebook-client-secret env)
   :redirect-uri (:bs-facebook-redirect-uri env)})

(s/defn ^:always-validate flickr :- Flickr
  "Return the Flickr config from `env`."
  [env]
  {:api-key (:bs-flickr-api-key env)
   :client-id (:bs-flickr-client-id env)
   :client-secret (:bs-flickr-client-secret env)})

(s/defn ^:always-validate google-maps :- GoogleMaps
  "Return the Google Maps config from `env`."
  [env]
  {:api-key (:bs-google-maps-api-key env)})

(s/defn ^:always-validate google :- Google
  "Return the config for Google services from `env`."
  [env]
  {:maps (google-maps env)})

(s/defn ^:always-validate reload :- Reload
  "Return the namespace reload config from `env`."
  [env]
  {:disabled (true? (edn/read-string (:bs-reload-disabled env)))
   :source-paths ["src" "test"]})

(s/defn ^:always-validate jws :- JWS
  "Return the JWS config from `env`."
  [env]
  {:secret (:bs-jws-secret env)
   :expires-in (or (edn/read-string (:bs-jws-expires-in env)) 3600)})

(s/defn ^:always-validate s3-backup :- S3Backups
  "Return the S3 backup config from `env`."
  [env]
  {:bucket (:bs-s3-backup-bucket env)
   :prefix (or (:bs-s3-backup-prefix env) "backups")})

(s/defn ^:always-validate s3-photo :- S3Photos
  "Return the S3 photo config from `env`."
  [env]
  {:bucket (:bs-s3-photo-bucket env)
   :prefix (or (:bs-s3-photo-prefix env) "photos")})

(s/defn ^:always-validate s3 :- S3
  "Return the S3 config from `env`."
  [env]
  {:backups (s3-backup env)
   :photos (s3-photo env)})

(s/defn ^:always-validate backup :- s/Any
  "Return the backup config from `env`."
  [env]
  {:aws (aws env)
   :db (db env)
   :s3 (s3-backup env)})

(s/defn ^:always-validate web :- Web
  "Return the web config from `env`."
  [env]
  {:api-client (api-client env)
   :bind-address (:bs-web-bind-address env)
   :bind-port (or (bind-port env :bs-web-bind-port) 20001)
   :google (google env)
   :facebook (facebook env)
   :figwheel (figwheel env)
   :flickr (flickr env)
   :ssl-redirect? (true? (edn/read-string (:bs-web-ssl-redirect env)))})

(s/defn ^:always-validate web-client :- WebClient
  "Return the web client config from `env`."
  [env]
  {:scheme (keyword (:bs-web-scheme env :scheme))
   :server-name (:bs-web-server-name env)
   :server-port (edn/read-string (:bs-web-server-port env))})

(s/defn ^:always-validate api :- Api
  "Return the API config from `env`."
  [env]
  {:api-client (api-client env)
   :bind-address (:bs-api-bind-address env)
   :bind-port (or (bind-port env :bs-api-bind-port) 20000)
   :broker (broker env)
   :db (db env)
   :google (google env)
   :facebook (facebook env)
   :flickr (flickr env)
   :jws (jws env)
   :reload (reload env)
   :ssl-redirect? (true? (edn/read-string (:bs-web-ssl-redirect env)))
   :web (web-client env)})

(s/defn ^:always-validate scheduler :- Scheduler
  "Return the API config from `env`."
  [env]
  {:broker (broker env)
   :db (db env)})

(s/defn ^:always-validate config :- Config
  "Return the system config from `env`."
  [env]
  {:api (api env)
   :api-client (api-client env)
   :aws (aws env)
   :broker (broker env)
   :db (db env)
   :facebook (facebook env)
   :flickr (flickr env)
   :google (google env)
   :reload (reload env)
   :web (web env)
   :web-client (web-client env)})
