(ns burningswell.config.core
  (:require [burningswell.config.schema :as schemas]
            [schema.core :as s]
            [clojure.edn :as edn]))

(s/defn ^:always-validate api :- schemas/Api
  "Return the API config from `env`."
  [env]
  {:bind-address (:bs-api-bind-address env)
   :bind-port (edn/read-string (:bs-api-bind-port env))})

(s/defn ^:always-validate api-client :- schemas/ApiClient
  "Return the API client config from `env`."
  [env]
  {:scheme (keyword (:bs-api-scheme env))
   :server-name (:bs-api-server-name env)
   :server-port (edn/read-string (:bs-api-server-port env))})

(s/defn ^:always-validate aws :- schemas/Aws
  "Return the AWS config from `env`."
  [env]
  {:access-key (:bs-aws-access-key env)
   :account-number (:bs-aws-account-number env)
   :secret-key (:bs-aws-secret-key env)})

(s/defn ^:always-validate broker :- schemas/Broker
  "Return the broker config from `env`."
  [env]
  {:password (:bs-broker-password env)
   :scheme (or (:bs-broker-scheme env) :amqp)
   :server-name (:bs-broker-server-name env)
   :server-port (edn/read-string (:bs-broker-server-port env))
   :username (:bs-broker-username env)
   :vhost (or (:bs-broker-vhost env) "/")})

(s/defn ^:always-validate db :- schemas/Db
  "Return the database config from `env`."
  [env]
  {:name (:bs-db-name env)
   :password (:bs-db-password env)
   :test (true? (edn/read-string (:bs-db-rollback env)))
   :scheme (keyword (:bs-db-scheme env))
   :server-name (:bs-db-server-name env)
   :server-port (edn/read-string (:bs-db-server-port env))
   :username (:bs-db-username env)})

(s/defn ^:always-validate facebook :- schemas/Facebook
  "Return the Facebook config from `env`."
  [env]
  {:client-id (:bs-facebook-client-id env)
   :client-secret (:bs-facebook-client-secret env)
   :redirect-uri (:bs-facebook-redirect-uri env)})

(s/defn ^:always-validate flickr :- schemas/Flickr
  "Return the Flickr config from `env`."
  [env]
  {:client-id (:bs-flickr-client-id env)
   :client-secret (:bs-flickr-client-secret env)})

(s/defn ^:always-validate google :- schemas/Google
  "Return the Google config from `env`."
  [env]
  {:api-key (:bs-google-api-key env)
   :client-id (:bs-google-client-id env)
   :client-secret (:bs-google-client-secret env)
   :redirect-uri (:bs-google-redirect-uri env)})

(s/defn ^:always-validate reload :- schemas/Reload
  "Return the namespace reload config from `env`."
  [env]
  {:disabled (true? (edn/read-string (:bs-reload-disabled env)))
   :source-paths ["src" "test"]})

(s/defn ^:always-validate web :- schemas/Web
  "Return the web config from `env`."
  [env]
  {:bind-address (:bs-web-bind-address env)
   :bind-port (edn/read-string (:bs-web-bind-port env))})

(s/defn ^:always-validate web-client :- schemas/WebClient
  "Return the web client config from `env`."
  [env]
  {:scheme (keyword (:bs-web-scheme env))
   :server-name (:bs-web-server-name env)
   :server-port (edn/read-string (:bs-web-server-port env))})

(s/defn ^:always-validate config :- schemas/Config
  "Return the system config from `env`."
  [env]
  {:api (api env)
   :api-client (api-client env)
   :aws (aws env)
   :broker (broker env)
   :db (db env)
   :facebook (facebook env)
   :flickr (flickr env)
   :google (google env)
   :reload (reload env)
   :web (web env)
   :web-client (web-client env)})
