(ns burningswell.api.airports
  "The airports in the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.airports :as airports]
            [burningswell.http.response :refer [created ok]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn airport-not-found
  "Return a 404 response for an airport that could not be found by `id`."
  [id]
  (not-found (format "Airport %s not found" id)))

(defn airports
  "List all airports."
  [{:keys [api-client db query-params]}]
  (let [airports (airports/all db query-params)]
    (ok (hal/links api-client :airport airports))))

(defn create-airport
  "Create a new airport."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateAirport data)
  (let [airport (airports/insert db data)
        airport (hal/link api-client :airport airport)]
    (publish broker "airports.created" airport)
    (created airport)))

(defn airport
  "Show an airport."
  [{:keys [api-client db path-params]}]
  (if-let [airport (airports/by-id db (:id path-params))]
    (ok (hal/link api-client :airport airport))
    (airport-not-found (:id path-params))))

(defn delete-airport
  "Delete an airport."
  [{:keys [broker db path-params]}]
  (if-let [airport (airports/by-id db (:id path-params))]
    (do (airports/delete db airport)
        (publish broker "airports.deleted" airport)
        (no-content))
    (airport-not-found (:id path-params))))

(defn update-airport
  "Update an airport."
  [{:keys [api-client data broker db path-params]}]
  (if-let [airport (airports/by-id db (:id path-params))]
    (do (s/validate CreateAirport data)
        (let [airport (airports/update db (merge airport data))
              airport (hal/link api-client :airport airport)]
          (publish broker "airports.updated" airport)
          (ok airport)))
    (airport-not-found (:id path-params))))

(set! *warn-on-reflection* false)
