(ns burningswell.api.spots
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.photos :refer [make-photo]]
            [burningswell.api.schemas :refer :all]
            [burningswell.api.spec.spot :as spot]
            [burningswell.api.validation :as validation]
            [burningswell.db.comments :as comments]
            [burningswell.db.countries :as countries]
            [burningswell.db.photos :as photos]
            [burningswell.db.ratings :as ratings]
            [burningswell.db.regions :as regions]
            [burningswell.db.sessions :as sessions]
            [burningswell.db.spots :as spots]
            [burningswell.db.time-zones :as time-zones]
            [burningswell.db.weather :as weather]
            [burningswell.http.response :refer [created ok]]
            [clj-time.coerce :refer [to-date-time to-long]]
            [clj-time.core :refer [hour]]
            [datumbazo.core :refer [with-connection]]
            [hiccup.core :refer [html]]
            [burningswell.api.spec.request :as request]))

(defn make-spot [api-client spot]
  (-> (hal/link api-client :spot spot)
      (update-in [:_embedded :photo]
                 #(make-photo api-client %))))

(defn enhance-spots
  "Enhance `spots` with a photo and the current weather."
  [api-client db spots & [opts]]
  (let [spots (spots/assoc-photo db spots)
        spots (spots/assoc-current-weather db spots opts)]
    (map #(make-spot api-client %) spots)))

(defn enhance-spot [api-client db spot & [opts]]
  (first (enhance-spots api-client db [spot] opts)))

(defapi spot
  "Return a surf spot."
  [{:keys [api-client db path-params query-params] :as request}]
  (when-let [spot (spots/by-id db (:id path-params))]
    (ok (enhance-spot api-client db spot query-params))))

(defapi spots
  "Return a list of surf spots."
  [{:keys [api-client db query-params]}]
  {:request ::request/spots}
  (let [spots (spots/all db query-params)
        spots (enhance-spots api-client db spots query-params)]
    (ok spots)))

(defapi create-spot
  "Create a new spot."
  [{:keys [api-client data broker db identity path-params]}]
  (validation/validate-errors! {:db db} ::spot/create-spot data)
  (let [location (:location data)
        data (-> (update-in
                  data [:_embedded :country :id]
                  #(or % (:id (countries/closest db location))))
                 (update-in [:_embedded :region :id]
                            #(or % (:id (regions/closest db location))))
                 (assoc-in [:_embedded :user] identity)
                 (assoc-in [:_embedded :time-zone]
                           (time-zones/by-location db location)))
        spot (spots/insert db data)
        spot (enhance-spot api-client db spot)]
    (publish broker "spots.created" spot)
    (created spot)))

(defapi delete-spot
  "Delete a surf spot."
  [{:keys [broker db path-params]}]
  (when-let [spot (spots/by-id db (:id path-params))]
    (spots/delete db spot)
    (publish broker "spots.deleted" spot)
    (no-content)))

(defapi update-spot
  "Update a spot."
  [{:keys [api-client data broker db identity path-params]}]
  (when-let [spot (spots/by-id db (:id path-params))]
    (let [data (-> (assoc data :id (:id spot))
                   (assoc-in [:_embedded :user] identity))]
      (validation/validate-errors! {:db db} ::spot/create-spot data)
      (->> (spots/update db data)
           (enhance-spot api-client db)
           (publish broker "regions.updated")
           (ok)))))

(defapi nearby-spots
  "List all spots nearby a spot."
  [{:keys [api-client db path-params query-params]}]
  (when-let [spot (spots/by-id db (:id path-params))]
    (let [spots (spots/nearby-spots db spot query-params)
          spots (enhance-spots api-client db spots query-params)]
      (ok spots))))

(defapi comments
  "List all comments of the spot."
  [{:keys [api-client db path-params query-params]}]
  (when-let [spot (spots/by-id db (:id path-params))]
    (let [comments (comments/by-spot db spot query-params)
          comments (hal/links api-client :comment comments)]
      (ok comments))))

(defapi photos
  "List all photos of the spot."
  [{:keys [api-client db path-params query-params]}]
  (when-let [spot (spots/by-id db (:id path-params))]
    (->> (photos/by-spot db spot query-params)
         (photos/assoc-images db)
         (map #(make-photo api-client %))
         (ok))))

(defapi ratings
  "List the ratings of a spot."
  [{:keys [api-client db path-params query-params]}]
  (when-let [spot (spots/by-id db (:id path-params))]
    (let [ratings (ratings/ratings-by-spot db spot query-params)]
      (ok ratings))))

(defapi sessions
  "List the sessions of a spot."
  [{:keys [api-client db path-params query-params]}]
  (when-let [spot (spots/by-id db (:id path-params))]
    (let [sessions (sessions/sessions-by-spot db spot query-params)]
      (ok sessions))))

(defapi weather
  "Show the the weather at a spot."
  [{:keys [api-client db path-params query-params]}]
  (if-let [spot (spots/by-id db (:id path-params))]
    (let [weather (weather/weather-forecast-by-spots db [spot] query-params)]
      (ok weather {"cache-control" "public, max-age=86400"}))))

(defn render-bar-chart
  [data]
  (let [timestamps (map :valid-time data)
        fill-color "#444444"
        values (map :value data)
        max-value (apply max (remove nil? values))
        max-width 500
        max-height 89
        padding 1.0
        width (/ max-width (count values))]
    (str
     "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\" "
     "\"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">"
     (html
      [:svg {:class "wave-heights-chart"
             :version "1.1"
             :preserveAspectRatio "none"
             :viewBox (format "0 0 %s %s" max-width max-height)
             :xmlns "http://www.w3.org/2000/svg"}
       [:g
        (map-indexed
         (fn [index data]
           (let [value (:value data)
                 timestamp (:valid-time data)
                 height (if value (/ (* max-height value) max-value) 1)
                 hours (hour (to-date-time timestamp))
                 class (if (and (>= hours 5) (<= hours 22)) "day" "night")]
             [:g
              {:class "wave-heights-chart__day"
               :data-time (to-long timestamp)
               :data-value value}
              [:rect
               {:class "wave-heights-chart__day-bg"
                :height max-height
                :style "fill: #cccccc; fill-opacity: 0;"
                :width (- width padding)
                :x (+ (* index width) padding)
                :y 0}]
              [:rect
               {:class "wave-heights-chart__day-height"
                :height height
                :style (str "fill: " fill-color ";")
                :width (- width padding)
                :x (+ (* index width) padding)
                :y (- max-height height)}]]))
         data)]]))))

(defapi wave-heights
  "Return the wave heights SVG chart."
  [{:keys [api-client db path-params query-params]}]
  (with-connection [db db]
    (when-let [spot (spots/by-id db (:id path-params))]
      (let [wave-heights (weather/wave-heights-by-spot db spot query-params)]
        (if (some :value wave-heights)
          (ok (render-bar-chart wave-heights)
              {"cache-control" "public, max-age=86400"
               "content-type" "image/svg+xml"})
          (not-found "No wave height available."))))))
