(ns burningswell.api.airports
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.airports :as airports]
            [burningswell.http.response :refer [created ok]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defapi airports
  "List all airports."
  [{:keys [api-client db query-params]}]
  (let [airports (airports/all db query-params)]
    (ok (hal/links api-client :airport airports))))

(defapi create-airport
  "Create a new airport."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateAirport data)
  (let [airport (airports/insert db data)
        airport (hal/link api-client :airport airport)]
    (publish broker "airports.created" airport)
    (created airport)))

(defapi airport
  "Show an airport."
  [{:keys [api-client db path-params]}]
  (when-let [airport (airports/by-id db (:id path-params))]
    (ok (hal/link api-client :airport airport))))

(defapi delete-airport
  "Delete an airport."
  [{:keys [broker db path-params]}]
  (when-let [airport (airports/by-id db (:id path-params))]
    (airports/delete db airport)
    (publish broker "airports.deleted" airport)
    (no-content)))

(defapi update-airport
  "Update an airport."
  [{:keys [api-client data broker db path-params]}]
  (when-let [airport (airports/by-id db (:id path-params))]
    (s/validate CreateAirport data)
    (let [airport (airports/update db (merge airport data))
          airport (hal/link api-client :airport airport)]
      (publish broker "airports.updated" airport)
      (ok airport))))

(set! *warn-on-reflection* false)
