(ns burningswell.api.photos
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.photos :as photos]
            [burningswell.http.response :refer [created ok]]
            [schema.core :as s]))

(defn make-photo [api-client photo]
  (if photo
    (hal/link api-client :photo photo)))

(defn- photo-not-found
  "Return a 404 response for a photo that could not be found by `id`."
  [id]
  (not-found (format "Photo %s not found" id)))

(defn photos
  "List all photos."
  [{:keys [api-client db query-params]}]
  (let [photos (photos/with-images db query-params)
        photos (hal/links api-client :photo photos)]
    (ok photos)))

(defn photo
  "Show a photo."
  [{:keys [api-client db identity path-params query-params]}]
  (if-let [photo (photos/with-images-by-id
                   db (:id path-params) {:user identity} query-params)]
    (let [photo (hal/link api-client :photo photo)]
      (ok photo))
    (photo-not-found (:id path-params))))

(defn create-photo
  "Create a new photo."
  [{:keys [api-client data broker db identity path-params]}]
  (s/validate CreatePhoto data)
  (let [data (assoc-in data [:_embedded :user] identity)
        photo (photos/insert db data)
        photo (hal/link api-client :photo photo)]
    (publish broker "photos.created" photo)
    (created photo)))

(defn photo-likes
  "Return the likes and dislikes of a photo."
  [{:keys [api-client data broker db path-params query-params]}]
  (if-let [photo (photos/by-id db (:id path-params) query-params)]
    (ok (photos/likes db photo))
    (photo-not-found (:id path-params))))

(defn like-photo
  "Like a photo."
  [{:keys [api-client broker db identity path-params]}]
  (if-let [photo (photos/by-id db (:id path-params))]
    (do (photos/like db photo identity)
        (->> (photos/with-images-by-id db (:id path-params) {:user identity})
             (hal/link api-client :photo)
             (ok )))
    (photo-not-found (:id path-params))))

(defn dislike-photo
  "Dislike a photo."
  [{:keys [api-client broker db identity path-params]}]
  (if-let [photo (photos/by-id db (:id path-params))]
    (do (photos/dislike db photo identity)
        (->> (photos/with-images-by-id db (:id path-params) {:user identity})
             (hal/link api-client :photo)
             (ok )))
    (photo-not-found (:id path-params))))
