(ns burningswell.api.validation
  (:require [clojure.spec :as s]
            [burningswell.db.users :as users]
            [burningswell.api.errors :as errors]
            [clojure.string :as str]))

(def ^:dynamic *context*
  "Dynamic var that can be bound to a context for specs that need
  state." nil)

(def email-regex
  "The regular expression for an email address."
  #"^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,63}$")

(defn max-length [len]
  #(<= (count %) len))

(defn min-length [len]
  #(>= (count %) len))

(defn email-available?
  "Returns true if `email` is available, otherwise false."
  [email]
  (users/email-available? (:db *context*) email))

(defn username-available?
  "Returns true if `username` is available, otherwise false."
  [username]
  (users/username-available? (:db *context*) username))

(defn email?
  "Returns true if `s` is a valid email address, otherwise false."
  [s]
  (re-matches email-regex s))

(defn units?
  "Returns true if `x` is a valid unit system, otherwise false."
  [x]
  (contains? #{"eu" "uk" "us"} x))

(defn not-blank? [s]
  (not (str/blank? s)))

(s/def ::email
  (s/and string?
         not-blank?
         email?
         email-available?))

(s/def ::username
  (s/and string?
         not-blank?
         (min-length 2)
         (max-length 32)
         username-available?))

(s/def ::password
  (s/and string?
         not-blank?
         (min-length 4)
         (max-length 1024)))

(s/def ::units
  (s/and string? units?))

(s/def ::signup
  (s/keys :req-un [::email ::username ::password]))

(s/def ::user-settings
  (s/keys :opt-un [::units]))

(defmacro with-context [context & body]
  `(binding [*context* ~context]
     ~@body))

(defn validate-data
  [system spec data]
  (with-context system
    (errors/explain-data spec data)))

(defn validate-errors
  [system spec data]
  (with-context system
    (errors/explain-errors spec data)))

(defn validate-errors!
  [system spec data]
  (with-context system
    (let [errors (errors/explain-errors spec data)]
      (if (empty? errors)
        data
        (throw (ex-info "Validation error"
                        {:type :validation-error
                         :data data
                         :errors errors}))))))
