(ns burningswell.api.errors
  (:require [burningswell.api.i18n :refer [t]]
            [clojure.spec :as s]))

(defn i18n-key
  "Return the I18n attribute key for `problem`."
  [problem]
  (-> problem :via last name keyword))

(defn dispatch-key
  "Return the dispatch key for  `problem`."
  [{:keys [pred] :as problem}]
  (cond
    (symbol? pred)
    pred
    (sequential? pred)
    (first pred)))

(defmulti error-path
  "Adds a :key-path to a problem."
  dispatch-key)

(defmethod error-path 'contains?
  [{:keys [path pred]}]
  (conj path (last pred)))

(defmethod error-path :default
  [{:keys [path] :as problem}]
  (if (empty? path)
    [(i18n-key problem)] path))

(defmulti error-msg
  "Returns a human readable string of a problem."
  dispatch-key)

(defmethod error-msg 'contains? [problem]
  (str (t :en (-> problem :pred last)) " "
       (t :en :contains?) "."))

(defmethod error-msg 'email? [problem]
  (str (t :en :invalid-email) "."))

(defmethod error-msg 'email-available? [problem]
  (str (t :en :email-available?) "."))

(defmethod error-msg 'max-length [problem]
  (let [[_ length] (:pred problem)]
    (str (t :en (i18n-key problem)) " "
         (t :en :max-length length) ".")))

(defmethod error-msg 'min-length [problem]
  (let [[_ length] (:pred problem)]
    (str (t :en (i18n-key problem)) " "
         (t :en :min-length length) ".")))

(defmethod error-msg 'not-blank? [problem]
  (str (t :en (i18n-key problem)) " "
       (t :en :not-blank?) "."))

(defmethod error-msg 'string? [problem]
  (str (t :en (i18n-key problem)) " "
       (t :en :not-string) "."))

(defmethod error-msg 'username-available? [problem]
  (str (t :en :username-available?) "."))

(defmethod error-msg 'units? [problem]
  (str (t :en :units?) "."))

(defmethod error-msg :default [problem]
  (str (t :en :unknown-error) "."))

(defn add-error
  "Add the human readable error to the `problem`."
  [problem]
  (->> {:message (error-msg problem)
        :path (error-path problem)}
       (assoc problem :error)))

(defn add-errors
  "Add human readable errors to the `problems`."
  [problems]
  (update problems ::s/problems #(map add-error %)))

(defn- extract-flat-errors
  "Extract only the errors from `problems`."
  [problems]
  (mapv #(-> % :error :message) (::s/problems problems)))

(defn- extract-nested-errors
  [problems]
  (reduce
   (fn [errors problem]
     (let [{:keys [message path]} (:error problem)]
       (update-in errors path #(conj (or % []) message))))
   {} (::s/problems problems)))

(defn errors
  "Extract only the errors from `problems`."
  [spec problems]
  (let [description (s/describe spec)]
    (cond
      (and (sequential? description)
           (= 'keys (first description)))
      (extract-nested-errors problems)
      :else (extract-flat-errors problems))))

(defn explain-data
  [spec data]
  (-> (s/explain-data spec data)
      (add-errors)))

(defn explain-errors
  [spec data]
  (->> (explain-data spec data)
       (errors spec )
       (not-empty)))
