(ns burningswell.api.roles
  "The roles of users."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.roles :as roles]
            [burningswell.http.response :refer [created ok]]
            [datumbazo.core :refer [with-connection]]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn role-not-found
  "Return a 404 response for a role that could not be found by `id`."
  [id]
  (not-found (format "Role %s not found" id)))

(defn roles
  "List all roles."
  [{:keys [api-client db query-params]}]
  (with-connection [db db]
    (let [roles (roles/roles db query-params)]
      (ok roles))))

(defn create-role
  "Create a new role."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateRole data)
  (with-connection [db db]
    (let [role (roles/create-role db data)]
      (publish broker "roles.created" role)
      (created role))))

(defn role
  "Show a role."
  [{:keys [api-client db path-params]}]
  (with-connection [db db]
    (if-let [role (roles/role-by-id db (:id path-params))]
      (ok role)
      (role-not-found (:id path-params)))))

(defn delete-role
  "Delete a role."
  [{:keys [broker db path-params]}]
  (with-connection [db db]
    (if-let [role (roles/role-by-id db (:id path-params))]
      (do (roles/delete-role db role)
          (publish broker "roles.deleted" role)
          (no-content))
      (role-not-found (:id path-params)))))

(defn update-role
  "Update a role."
  [{:keys [api-client data broker db path-params]}]
  (s/validate CreateRole data)
  (with-connection [db db]
    (if-let [role (roles/role-by-id db (:id path-params))]
      (let [role (roles/update-role db (merge role data))]
        (publish broker "roles.updated" role)
        (ok role))
      (role-not-found (:id path-params)))))

(set! *warn-on-reflection* false)
