(ns burningswell.api.authentication
  (:require [buddy.auth.backends.httpbasic :refer [http-basic-backend]]
            [buddy.auth.backends.token :refer [jws-backend]]
            [buddy.auth.middleware :as middleware]
            [burningswell.db.users :as users]
            [io.pedestal.interceptor :refer [interceptor]]))

(defn- http-basic-auth-fn
  "The HTTP basic authentication fn."
  [request credentials]
  (users/authenticate (:db request) credentials))

(defn- wrap-identity
  "Make sure that each request has an :identity key."
  [handler]
  (fn [request]
    (handler (update-in request [:identity] identity))))

(defn wrap-authentication
  "Wrap `handler` with authentication."
  [handler resources]
  (let [jwt-secret (-> resources :config :jwt :secret)]
    (-> (wrap-identity handler)
        (middleware/wrap-authentication
         (http-basic-backend
          {:realm "Burning Swell API"
           :authfn http-basic-auth-fn})
         (jws-backend {:secret jwt-secret})))))

(def authentication-interceptor
  (interceptor
   {:name ::authentication-interceptor
    :enter
    (fn [context]
      (let [jwt-secret (-> context :request :config :jwt :secret)
            response (middleware/authentication-request
                      (:request context)
                      (http-basic-backend
                       {:realm "Burning Swell API"
                        :authfn http-basic-auth-fn})
                      (jws-backend {:secret jwt-secret}))]
        (assoc-in context [:request :identity] (:identity response))))}))
