(ns burningswell.api.users
  (:require [alumbra.claro :as claro]
            [burningswell.api.core :as core]
            [burningswell.api.middleware.authentication :as auth]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.db.favourites :as favourites]
            [burningswell.db.users :as users]
            [claro.access :as access]
            [claro.data :as data]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]))

(s/def :burningswell.api.users/params
  (s/keys :opt-un [:burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.specs/location
                   :burningswell.api.specs/direction
                   :burningswell.api.specs/distance
                   :burningswell.api.search/query
                   :burningswell.api.users/sort]))

(s/def :burningswell.api.users.favourites/params
  (s/keys :opt-un [:burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.specs/direction
                   :burningswell.api.favourites/sort]))

(defrecord Favourites [after before direction distance
                      first last sort types]
  conform/Params
  (conform [params env]
    :burningswell.api.users.favourites/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} users]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :favourites params)
          :types types}
         (favourites/resolve-user-favourites db users))))

(defrecord UserById [id]
  claro/GraphQL
  (__typename [_ _] "User")

  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :user :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} users]
    (users/select-batch db users)))

(defrecord UserByUsername [username]
  claro/GraphQL
  (__typename [_ _] "User")

  identifier/Identifier
  (identifier [_ _]
    {:type :user :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} users]
    (users/select-batch db users {:join {:username :citext}})))

(defrecord User [id username]
  data/Resolvable
  data/PureResolvable
  (resolve! [resolvable env]
    (cond
      (number? id)
      (map->UserById {:id id})
      (string? id)
      (map->UserById (core/parse-id id))
      username
      (map->UserByUsername {:username username})
      :else nil)))

(defrecord Users [after before direction distance
                  first last location sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.users/params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:direction direction
          :distance distance
          :limit (core/limit params)
          :location location
          :offset (core/offset params)
          :query query
          :sort sort}
         (users/search db))))

(defrecord Emails [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.emails/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} users]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :emails params)}
         (sql/has-many db users :users :emails))))

(defrecord Roles [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.roles/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} users]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :roles params)}
         (sql/has-and-belongs-to-many db users :users :roles))))

(defrecord ProfilePhotos [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.photos/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} users]
    (->> {:join-table :user-profile-photos
          :limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :photos params)}
         (sql/has-and-belongs-to-many db users :users :photos))))

(defrecord PrivateUser [id]
  access/Read
  (can-read? [params env user]
    (or (auth/has-role? (-> env :user) :admin)
        (= (:id user) (-> env :user :id))))

  data/Resolvable
  data/PureResolvable
  (resolve! [resolvable _]
    (map->User resolvable)))
