(ns burningswell.api.regions
  (:require [burningswell.api.core :as core]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.api.specs :as specs]
            [burningswell.api.validation :as v]
            [burningswell.api.weather :as weather]
            [burningswell.db.favourites :as favourites]
            [burningswell.db.regions :as regions]
            [claro.data :as data]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]))

(s/def :burningswell.api.regions/id
  (s/and ::specs/id (v/db-id-exists? :regions)))

(s/def :burningswell.api.regions/except
  (s/nilable (s/coll-of :burningswell.api.regions/id)))

(s/def :burningswell.api.regions/countries
  (s/nilable (s/coll-of :burningswell.api.countries/id)))

(s/def :burningswell.api.regions/params
  (s/keys :opt-un [:burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.regions/countries
                   :burningswell.api.regions/except
                   :burningswell.api.regions/sort
                   :burningswell.api.search/query
                   :burningswell.api.specs/direction
                   :burningswell.api.specs/distance
                   :burningswell.api.specs/location
                   :burningswell.api.specs/min-spots]))

(defrecord Favourite [id]
  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db user]} regions]
    (favourites/resolve-favourite db user :region regions)))

(defrecord Region [id name country]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :region
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} regions]
    (regions/select-batch db regions {:except [:geom]})))

(defrecord Regions [after before countries first direction distance except
                    last location min-spots sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.regions/params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:countries countries
          :direction direction
          :distance distance
          :except except
          :limit (core/limit params)
          :location location
          :min-spots min-spots
          :offset (core/offset params)
          :query query
          :sort sort}
         (regions/search db))))

(defrecord Airports [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.airports/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} regions]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :airports params)}
         (sql/has-many db regions :regions :airports))))

(defrecord Spots [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.spots/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} regions]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :spots params)}
         (sql/has-many db regions :regions :search.spots))))

(defrecord Photos [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.photos/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} regions]
    (->> {:join-table :photos-regions
          :limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :photos params)}
         (sql/has-and-belongs-to-many
          db regions :regions :photos))))

(defrecord Ports [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.ports/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} regions]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :ports params)}
         (sql/has-many db regions :regions :ports))))

(weather/define-weather-resolvers :regions :weather.regions)
