(ns burningswell.api.jwt
  (:require [buddy.sign.jwt :as jwt]
            [clojure.spec.alpha :as s]
            [clj-time.core :as t]))

(defrecord JWT [secret expires-in])

(s/def ::secret string?)
(s/def ::expires-in pos-int?)

(s/def ::jwt
  (s/keys :req-un [::secret ::expires-in]))

(defn sign
  "Encode `claims` as a JWT token."
  [jwt claims & [opts]]
  (jwt/sign claims (:secret jwt) opts))

(s/fdef sign
  :args (s/cat :jwt ::jwt :claims map? :opts (s/? (s/nilable map?))))

(defn unsign
  "Decode `token` as a JWT token."
  [jwt token]
  (jwt/unsign token (:secret jwt)))

(s/fdef unsign
  :args (s/cat :jwt ::jwt ::token string?))

(defn auth-token
  "Returns a JWT authentication token for `user`."
  [jwt user]
  (sign jwt (select-keys user [:id])
        {:exp (t/plus (t/now) (t/seconds (:expires-in jwt)))}))

(s/fdef auth-token
  :args (s/cat :jwt ::jwt ::user map?))

(defn jwt
  "Returns a JWT component."
  [config]
  {:pre [(s/valid? ::jwt config)]}
  (map->JWT config))

(s/fdef jwt
  :args (s/cat :config ::jwt))

(comment
  (prn (auth-token (:jwt reloaded.repl/system) {:id 1})))
