(ns burningswell.api.weather
  (:require [burningswell.api.core :as core]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.pagination :as p]
            [burningswell.api.weather.models :refer [map->Model]]
            [burningswell.api.weather.variables :refer [map->Variable]]
            [burningswell.db.weather.variables :as variables]
            [claro.data :as data]
            [clj-time.core :as t]
            [clj-time.coerce :refer [to-date-time]]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [inflections.core :as infl]
            [manifold.deferred :as d]))

(defrecord Weather [location]
  data/Resolvable
  (resolve! [params env]
    {:location location}))

(s/def ::end (s/nilable inst?))
(s/def ::start (s/nilable inst?))

(s/def ::variable-params
  (s/keys :opt-un [::end ::start]))

(defn resolve-variables [db table batch & [opts]]
  (d/future (map (fn [batch resolved]
                   (merge {} batch resolved))
                 batch (core/resolve-by-id db table batch {:type :bigint}))))

(defn select-variables [db variable-name batch source target & [opts]]
  (let [variable (variables/by-name
                  db (name (variables/keywords variable-name)))
        start (to-date-time (or (:start opts) (t/now)))
        end (or (to-date-time (:end opts)) (t/plus start (t/hours 3)))]
    (assert variable (str "Invalid long variable name: " variable-name))
    (->> {:limit (or (core/limit opts) 1)
          :offset (core/offset opts)
          :where (sql/where
                  `(and (= :variable-id ~(:id variable))
                        (>= :valid-time ~start)
                        (< :valid-time ~end)))
          :source-type :bigint
          :order-by '(order-by :valid-time)}
         (sql/has-many db batch source target)
         (map #(map (fn [x] (merge (select-keys variable [:unit]) x)) %))
         (d/future))))

(defn- record-singular [variable]
  (-> variable name infl/camel-case symbol))

(defn- record-plural [variable]
  (-> variable name infl/plural infl/camel-case symbol))

(defn- record-map-constructor [variable]
  (symbol (str "map->" (record-singular variable))))

(defn- query-map-constructor [variable]
  (symbol (str "map->" (record-plural variable))))

(defmacro define-record-resolver [variable target-table]
  `(defrecord ~(record-singular variable) [~'id]
     data/Resolvable
     data/BatchedResolvable
     (resolve-batch! [~'params ~'env ~'batch]
       (resolve-variables (:db ~'env) ~target-table ~'batch))
     data/Transform
     (transform [~'params ~'record]
       (-> ~'record
           (assoc :variable (map->Variable {:id (:variable-id ~'record)}))
           (assoc :model (map->Model {:id (:model-id ~'record)}))))))

(defmacro define-query-resolver [variable source-table target-table]
  `(defrecord ~(record-plural variable)
       [~'after ~'before ~'first ~'last ~'id ~'start ~'end]
     conform/Params
     (conform [~'params ~'env]
       :burningswell.api.weather/variable-params)

     data/Resolvable
     data/BatchedResolvable
     (resolve-batch! [~'params ~'env ~'batch]
       (select-variables
        (:db ~'env) ~variable ~'batch ~source-table ~target-table ~'params ))

     data/Transform
     (transform [~'params ~'batch]
       (p/->Connection ~'batch ~(record-map-constructor variable) ~'params))))

(defmacro define-weather [variables]
  `(defrecord ~'Weather [~'id]
     data/Resolvable
     data/PureResolvable
     (resolve! [~'params ~'env] {:id ~'id})
     data/Transform
     (transform [~'resolvable ~'resolve-result]
       (-> ~'resolve-result
           ~@(for [variable variables]
               `(assoc ~(-> variable infl/plural keyword)
                       (~(query-map-constructor variable)
                        ~'resolve-result )))))))

(defmacro define-weather-resolvers [source-table target-table]
  (let [variables (keys variables/keywords)]
    `(do ~@(for [variable variables]
             `(do (define-record-resolver
                    ~variable ~target-table)
                  (define-query-resolver
                    ~variable ~source-table ~target-table)))
         (define-weather ~variables))))
