(ns burningswell.api.oauth.providers
  (:require [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.db.oauth.providers :as providers]
            [burningswell.routes :as routes]
            [claro.data :as data]
            [clojure.spec.alpha :as s]
            [no.en.core :refer [format-url]]
            [oauth.one :as one]))

(defn- api-url
  "Returns a formatted API url for `uri`."
  [client uri]
  (format-url
   {:scheme (:scheme client)
    :server-name (:server-name client)
    :server-port (:server-port client)
    :uri uri}))

(defn- callback-url
  "Returns the API url to which the user gets redirected to from the
  OAuth provider."
  [client provider]
  (->> (routes/path :oauth-callback {:name (:name provider)})
       (api-url client)))

(defn- connect-url
  "Returns the API url to connect with an OAuth provider."
  [client provider]
  (->> (routes/path :oauth-connect {:name (:name provider)})
       (api-url client)))

(defn oauth-v1-consumer
  "Returns an OQuth v1 consumer for `provider`."
  [env provider]
  (let [key (-> provider :name keyword)]
    (one/make-consumer
     {:access-uri (:access-token-url provider)
      :authorize-uri (:authorization-url provider)
      :callback-uri (:callback-url provider)
      :key (-> env key :client-id)
      :request-uri (:request-token-url provider)
      :secret (-> env key :client-secret)
      :signature-algo :hmac-sha1})))

(defmulti enhance-provider
  "Enhance the OAuth provider with additional information."
  (fn [env provider] (-> provider :name keyword)))

(defmethod enhance-provider :google [{:keys [api-client] :as env} provider]
  (->> {:access-token-url "https://www.googleapis.com/oauth2/v4/token"
        :authorization-url "https://accounts.google.com/o/oauth2/v2/auth"
        :callback-url (callback-url api-client provider)
        :connect-url (connect-url api-client provider)}
       (merge provider)))

(defmethod enhance-provider :facebook [{:keys [api-client] :as env} provider]
  (->> {:access-token-url "https://graph.facebook.com/v2.10/oauth/access_token"
        :authorization-url "https://www.facebook.com/v2.10/dialog/oauth"
        :callback-url (callback-url api-client provider)
        :connect-url (connect-url api-client provider)}
       (merge provider)))

(defmethod enhance-provider :linkedin [{:keys [api-client] :as env} provider]
  (->> {:access-token-url "https://www.linkedin.com/oauth/v2/accessToken"
        :authorization-url "https://www.linkedin.com/oauth/v2/authorization"
        :callback-url (callback-url api-client provider)
        :connect-url (connect-url api-client provider)}
       (merge provider)))

(defmethod enhance-provider :twitter [{:keys [api-client] :as env} provider]
  (->> {:access-token-url "https://api.twitter.com/oauth/access_token"
        :authorization-url "https://api.twitter.com/oauth/authenticate"
        :callback-url (callback-url api-client provider)
        :connect-url (connect-url api-client provider)
        :request-token-url "https://api.twitter.com/oauth/request_token"}
       (merge provider)))

(defn by-name
  "Returns the OAuth provider by `name`."
  [env name]
  (some->> (providers/by-name (:db env) name)
           (enhance-provider env)))

(s/def :burningswell.api.oauth.providers/params
  (s/keys :opt-un [:burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.search/query
                   :burningswell.api.specs/direction]))

(defrecord Provider [id]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :oauth-provider
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db] :as env} providers]
    (map (partial enhance-provider env) (providers/select-batch db providers))))

(defrecord Providers [after before direction first last sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.oauth.providers/params)

  data/Resolvable
  (resolve! [_ {:keys [db]}]
    (providers/search db)))
