(ns burningswell.api.favourites
  (:require [burningswell.api.middleware.authentication :as auth]
            [burningswell.api.middleware.commands :as commands]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.events :as events]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.api.specs :as specs]
            [burningswell.api.validation :as v]
            [burningswell.db.favourites :as favourites]
            [claro.access :as access]
            [claro.data :as data]
            [clojure.spec.alpha :as s]))

(s/def :burningswell.api.favourites/id
  (s/and ::specs/id (v/db-id-exists? :favourites)))

;; Create

(s/def :burningswell.api.favourites.create.continent/input
  (s/and (s/keys :req-un [:burningswell.api.continents/id])
         (v/not-favourite? :continent)))

(s/def :burningswell.api.favourites.create.country/input
  (s/and (s/keys :req-un [:burningswell.api.countries/id])
         (v/not-favourite? :country)))

(s/def :burningswell.api.favourites.create.region/input
  (s/and (s/keys :req-un [:burningswell.api.regions/id])
         (v/not-favourite? :region)))

(s/def :burningswell.api.favourites.create.spot/input
  (s/and (s/keys :req-un [:burningswell.api.spots/id])
         (v/not-favourite? :spot)))

(defmulti create-spec
  "Returns the spec to create a favourite."
  (fn [params] (-> params :input :type)))

(defmethod create-spec :continent [_]
  (s/keys :req-un [:burningswell.api.favourites.create.continent/input]))

(defmethod create-spec :country [_]
  (s/keys :req-un [:burningswell.api.favourites.create.country/input]))

(defmethod create-spec :region [_]
  (s/keys :req-un [:burningswell.api.favourites.create.region/input]))

(defmethod create-spec :spot [_]
  (s/keys :req-un [:burningswell.api.favourites.create.spot/input]))

(defmethod create-spec :default [params]
  (throw (ex-info (str "Unknown favourite type: " (pr-str (-> params :input :type)))
                  {:params params})))

(defmulti create!
  "Create a favourite."
  (fn [db author params] (-> params :input :type)))

(defmethod create! :continent [db author params]
  (favourites/create-continent! db author (:input params)))

(defmethod create! :country [db author params]
  (favourites/create-country! db author (:input params)))

(defmethod create! :region [db author params]
  (favourites/create-region! db author (:input params)))

(defmethod create! :spot [db author params]
  (favourites/create-spot! db author (:input params)))

(defrecord Create [input]
  access/Resolve
  (can-resolve? [params {:keys [user]}]
    (auth/is-authenticated user))

  commands/Command
  (command [resolvable env]
    {:name :burningswell.api.commands/create-favourite})

  conform/Params
  (conform [params env]
    (create-spec params))

  data/Mutation
  data/Resolvable
  (resolve! [params {:keys [db user]}]
    (create! db user params))

  events/Events
  (events [resolvable {:keys [user]} favourite]
    [{:name :burningswell.api.events/favourite-created
      :favourite-id (:id favourite)}]))

;; Delete

(s/def :burningswell.api.favourites.delete/input
  (s/keys :req-un [:burningswell.api.favourites/id]))

(s/def :burningswell.api.favourites/delete
  (s/keys :req-un [:burningswell.api.favourites.delete/input]))

(defrecord Delete [input]
  access/Resolve
  (can-resolve? [params {:keys [db user]}]
    (or (auth/is-authenticated user)
        (v/is-favourite-author db input user)
        true))

  commands/Command
  (command [resolvable env]
    {:name :burningswell.api.commands/delete-favourite})

  conform/Params
  (conform [params env]
    :burningswell.api.favourites/delete)

  data/Mutation
  data/Resolvable
  (resolve! [params {:keys [db]}]
    (let [favourite (favourites/by-id db (:id input))]
      {:favourite (favourites/delete! db favourite)
       :node (favourites/target db favourite)}))

  events/Events
  (events [resolvable {:keys [user]} {:keys [favourite]}]
    [{:name :burningswell.api.events/favourite-deleted
      :favourite-id (:id favourite)}]))

(defrecord Favourite [id]
  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} favourites]
    (favourites/resolve-batch db :favourites favourites)))

;; Continent favourites

(defrecord ContinentFavourite [id]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :continent-favourite
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} favourites]
    (favourites/resolve-batch db :favourites.continents favourites)))

;; Country favourites

(defrecord CountryFavourite [id]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :country-favourite
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} favourites]
    (favourites/resolve-batch db :favourites.countries favourites)))

;; Region favourites

(defrecord RegionFavourite [id]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :region-favourite
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} favourites]
    (favourites/resolve-batch db :favourites.regions favourites)))

;; Spot favourites

(defrecord SpotFavourite [id]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :spot-favourite
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} favourites]
    (favourites/resolve-batch db :favourites.spots favourites)))
