(ns burningswell.api.continents
  (:require [burningswell.api.core :as core]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.api.specs :as specs]
            [burningswell.api.validation :as v]
            [burningswell.api.weather :as weather]
            [burningswell.db.favourites :as favourites]
            [burningswell.db.continents :as continents]
            [claro.data :as data]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]))

(s/def :burningswell.api.continents/id
  (s/and ::specs/id (v/db-id-exists? :continents)))

(s/def :burningswell.api.continents/except
  (s/nilable (s/coll-of :burningswell.api.continents/id)))

(s/def :burningswell.api.continents/params
  (s/keys :opt-un [:burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.specs/location
                   :burningswell.api.specs/direction
                   :burningswell.api.specs/distance
                   :burningswell.api.search/query
                   :burningswell.api.continents/except
                   :burningswell.api.continents/sort]))

(defrecord Favourite [id]
  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db user]} continents]
    (favourites/resolve-favourite db user :continent continents)))

(defrecord Continent [id]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :continent
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} continents]
    (continents/select-batch db continents {:except [:geom]})))

(defrecord Continents [after before direction distance except
                       first last location sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.continents/params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:direction direction
          :distance distance
          :except except
          :limit (core/limit params)
          :location location
          :offset (core/offset params)
          :query query
          :sort sort}
         (continents/search db))))

(defrecord Countries [after before direction first last location id min-spots sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.countries/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} continents]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :countries params)
          :where (sql/where `(>= :countries.spot-count ~min-spots))}
         (sql/has-many db continents :continents :search.countries))))

(defrecord Photos [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.photos/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} continents]
    (->> {:join-table :photos-continents
          :limit (core/limit params)
          :offset (core/offset params)
          :order-by (core/order-by :photos params)}
         (sql/has-and-belongs-to-many
          db continents :continents :photos))))

(weather/define-weather-resolvers :continents :weather.continents)
