(ns burningswell.api.weather.variables
  (:require [burningswell.api.core :as core]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.api.specs :as specs]
            [burningswell.db.util :refer [fulltext]]
            [burningswell.db.weather.variables :as variables]
            [claro.data :as data]
            [clj-time.core :as t]
            [clojure.spec.alpha :as s]
            [datumbazo.core :as sql]
            [geo.postgis :as geo]
            [manifold.deferred :as d]))

(s/def :burningswell.api.weather.variables/params
  (s/keys :opt-un [:burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.specs/direction
                   :burningswell.api.search/query
                   :burningswell.api.weather.variables/sort]))

(defrecord Variable [id name description]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :weather-variable
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} weather-variables]
    (variables/select-batch db weather-variables)))

(defrecord Variables [after before direction first last sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.weather.variables/params)

  data/Resolvable
  (resolve! [_ {:keys [db]}]
    @(sql/select db [:variables.id]
       (sql/from :weather.variables)
       (fulltext query :name :description)
       (sql/limit first)
       (sql/order-by (keyword (or sort :name))))))

(defrecord Models [after before direction first last id sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.weather.models/params)

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [params {:keys [db]} models]
    (->> {:join-table :weather.models-variables
          :limit (core/limit params)
          :offset (core/offset params)
          :order-by '(order-by :models.name)}
         (sql/has-and-belongs-to-many
          db models :weather.variables :weather.models))))

(def default-location
  (geo/point 4326 -2.69500976358495 43.40921730553149))

(defn- select-variable
  "Select the weather data for `variable`."
  [db variable location & [{:keys [start end limit]}]]
  (let [start (or start (t/minus (t/now) (t/hours 1)))
        end (or end (t/plus start (t/days 7)))]
    @(sql/select db [:model-id
                     :variable-id
                     :variables.unit
                     (sql/as location :location)
                     :valid-time
                     :reference-time
                     (sql/as `(cast
                               (avg_neighborhood
                                (st_neighborhood
                                 :rast 1 (st_transform ~location 3395)
                                 1 1))
                               :double-precision) :value)]
       (sql/from :weather.rasters)
       (sql/join :weather.variables.id :weather.rasters.variable-id)
       (sql/where `(and (st_intersects :rast (st_transform ~location 3395))
                        (= :variable-id ~(:id variable))
                        (>= :valid-time ~start)
                        (<= :valid-time ~end)))
       (sql/order-by :valid-time)
       (when limit (sql/limit limit)))))

;; Wave Heights

(s/def ::end (s/nilable inst?))
(s/def ::start (s/nilable inst?))
(s/def ::location (s/nilable ::specs/location))

(s/def ::variable-params
  (s/keys :opt-un [::end ::start ::location]))

(defrecord WaveHeights [after before first last location start end]
  conform/Params
  (conform [params env]
    ::variable-params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :start start
          :end end}
         (select-variable db {:id 3} (or location default-location))
         (d/future))))

;; Wind Directions

(defrecord WindDirections [after before first last location start end]
  conform/Params
  (conform [params env]
    ::variable-params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :start start
          :end end}
         (select-variable db {:id 10} (or location default-location))
         (d/future))))

;; Wind Speeds

(defrecord WindSpeeds [after before first last location start end]
  conform/Params
  (conform [params env]
    ::variable-params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:limit (core/limit params)
          :offset (core/offset params)
          :start start
          :end end}
         (select-variable db {:id 11} (or location default-location))
         (d/future))))
