(ns burningswell.api.signup
  (:require [burningswell.api.core :as core]
            [burningswell.api.gen :as gens]
            [burningswell.api.jwt :as jwt]
            [burningswell.api.middleware.commands :as commands]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.events :as events]
            [burningswell.api.validation :as v]
            [burningswell.db.signup :as signup]
            [claro.data :as data]
            [clojure.spec.alpha :as s]))

;; Input

(s/def :burningswell.api.signup.input/email
  (s/with-gen (s/and string? v/email-valid? v/email-available?)
    (constantly gens/email)))

(s/def :burningswell.api.signup.input/username
  (s/and string? (v/min-length 1) v/username-available?))

(s/def :burningswell.api.signup.input/password
  (s/and string? (v/min-length 5)))

(s/def :burningswell.api.signup/input
  (s/keys :req-un [:burningswell.api.signup.input/email
                   :burningswell.api.signup.input/username
                   :burningswell.api.signup.input/password]))

(s/def :burningswell.api/signup
  (s/keys :req-un [:burningswell.api.signup/input]))

(defn signup!
  "Sign up a new user."
  [env input]
  (let [user (signup/signup! (:db env) input)]
    {:auth-token (jwt/auth-token (:jwt env) user)
     :user user}))

(defrecord Create [input]
  commands/Command
  (command [resolvable env]
    {:name :burningswell.api.commands/signup
     :params (update-in resolvable [:input :password] core/munge-password)})

  conform/Params
  (conform [params env]
    :burningswell.api/signup)

  data/Mutation
  data/Resolvable
  (resolve! [params env]
    (signup! env input))

  events/Events
  (events [resolvable env result]
    (if (-> result :auth-token)
      [{:name :burningswell.api.events/user-created
        :user-id (-> result :user :id)}
       {:name :burningswell.api.events/signup-succeeded
        :user-id (-> result :user :id)}]
      [{:name :burningswell.api.events/signup-failed}])))

(defrecord Validate [input]
  conform/Params
  (conform [params env]
    :burningswell.api/signup)

  data/Mutation
  data/Resolvable
  (resolve! [params env]
    (core/valid? env :burningswell.api/signup params)))
